import os
import torch
import torch.nn.functional as F
from torch.optim import Adam
from sac.utils import soft_update, hard_update
from sac.model import GaussianPolicy, QNetwork, DeterministicPolicy
from torch.distributions.normal import Normal

class SAC(object):
    def __init__(self, num_inputs, action_space, args, logger=None):

        self.gamma = args.gamma
        self.tau = args.tau
        self.alpha = args.alpha

        self.policy_type = args.policy
        self.target_update_interval = args.target_update_interval
        self.automatic_entropy_tuning = args.automatic_entropy_tuning

        self.device = torch.device("cuda")

        self.critic = QNetwork(num_inputs, action_space.shape[0], args.hidden_size).to(device=self.device)
        self.critic_optim = Adam(self.critic.parameters(), lr=args.lr)

        self.critic_target = QNetwork(num_inputs, action_space.shape[0], args.hidden_size).to(self.device)
        hard_update(self.critic_target, self.critic)

        if self.policy_type == "Gaussian":
            # Target Entropy = −dim(A) (e.g. , -6 for HalfCheetah-v2) as given in the paper
            if self.automatic_entropy_tuning == True:
                self.target_entropy = args.kl#-torch.prod(torch.Tensor(action_space.shape).to(self.device)).item()
                self.log_alpha = torch.zeros(1, requires_grad=True, device=self.device)
                self.alpha_optim = Adam([self.log_alpha], lr=args.lr)

            self.policy = GaussianPolicy(num_inputs, action_space.shape[0], args.hidden_size, action_space).to(self.device)
            self.policy_optim = Adam(self.policy.parameters(), lr=args.lr)

        else:
            self.alpha = 0
            self.automatic_entropy_tuning = False
            self.policy = DeterministicPolicy(num_inputs, action_space.shape[0], args.hidden_size, action_space).to(self.device)
            self.policy_optim = Adam(self.policy.parameters(), lr=args.lr)
        self.logger = logger

        # lr decay
        num_steps = 50000
        self.actor_lr_scheduler = torch.optim.lr_scheduler.LambdaLR(optimizer=self.policy_optim,
                                                                     lr_lambda=lambda itr: (num_steps - itr) / num_steps)  # Step once per itr.
        self.critic_lr_scheduler = torch.optim.lr_scheduler.LambdaLR(optimizer=self.critic_optim,
                                                                     lr_lambda=lambda itr: (num_steps - itr) / num_steps)  # Step once per itr.
        self.alpha_lr_scheduler = torch.optim.lr_scheduler.LambdaLR(optimizer=self.alpha_optim,
                                                                        lr_lambda=lambda itr: (num_steps - itr) /num_steps)

    def setup_prior(self, model):
        self.prior = model

    def select_action(self, state, eval=False):
        state = torch.FloatTensor(state).to(self.device).unsqueeze(0)
        if eval == False:
            action, _, _ = self.policy.sample(state)
        else:
            #_, _, action = self.policy.sample(state)
            with torch.no_grad():
                action, _ = self.policy(state)
        return action.detach().cpu().numpy()[0]

    def update_parameters(self, memory, batch_size, updates, step):
        # Sample a batch from memory
        # state_batch, action_batch, reward_batch, next_state_batch, mask_batch = memory.sample(batch_size=batch_size)
        state_batch, action_batch, reward_batch, next_state_batch, mask_batch = memory

        state_batch = torch.FloatTensor(state_batch).to(self.device)
        next_state_batch = torch.FloatTensor(next_state_batch).to(self.device)
        action_batch = torch.FloatTensor(action_batch).to(self.device)
        reward_batch = torch.FloatTensor(reward_batch).to(self.device).unsqueeze(1)
        mask_batch = torch.FloatTensor(mask_batch).to(self.device).unsqueeze(1)

        with torch.no_grad():
           # print("NEXT STATE BATCH SHAPE", next_state_batch.shape)
            next_state_action, next_state_log_pi, next_state_mean, next_state_std = self.policy.sample(next_state_batch, True)
            qf1_next_target, qf2_next_target = self.critic_target(next_state_batch, next_state_action)
            #min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) - self.alpha * next_state_log_pi
            next_state_dist = Normal(next_state_mean, next_state_std)
            prior_dist = self.prior.dist(next_state_batch)
            next_kl = self.kl_divergence(next_state_dist, prior_dist)
            min_qf_next_target = torch.min(qf1_next_target, qf2_next_target) \
                                            - self.alpha * torch.clamp(next_kl, -100, 100)
                                            #* F.kl_div(next_state_action, self.prior.sample(next_state_batch))
            #print("K SHAPES AND THINGS", self.mc_kl_divergence(next_state_dist, self.prior.dist(next_state_batch)).shape)                                
            next_q_value = reward_batch + mask_batch * self.gamma * (min_qf_next_target)

        qf1, qf2 = self.critic(state_batch, action_batch)  # Two Q-functions to mitigate positive bias in the policy improvement step
        # assert not torch.isnan(next_state_mean.mean())
        # assert not torch.isnan(next_state_std.mean())
        # assert not torch.isnan(prior_dist.loc.mean())
        # assert not torch.isnan(prior_dist.scale.mean())
        # assert not torch.isnan(next_kl.mean())
        try:
            qf1_loss = F.mse_loss(qf1, next_q_value) # JQ = 𝔼(st,at)~D[0.5(Q1(st,at) - r(st,at) - γ(𝔼st+1~p[V(st+1)]))^2]
            qf2_loss = F.mse_loss(qf2, next_q_value) # JQ = 𝔼(st,at)~D[0.5(Q1(st,at) - r(st,at) - γ(𝔼st+1~p[V(st+1)]))^2]
        except:
            return 0, 0, 0, 0, 0
        pi, log_pi, pi_mean, pi_std = self.policy.sample(state_batch, True)

        qf1_pi, qf2_pi = self.critic(state_batch, pi)
        min_qf_pi = torch.min(qf1_pi, qf2_pi)
        prior = self.prior.dist(state_batch)
        #policy_loss = ((self.alpha * log_pi) - min_qf_pi).mean() # Jπ = 𝔼st∼D,εt∼N[α * logπ(f(εt;st)|st) − Q(st,f(εt;st))]
        pi_dist = Normal(pi_mean, pi_std)
        #print("SHAPES", pi_mean.shape, pi_std.shape, prior.loc.shape)
        kl = torch.clamp(self.kl_divergence(pi_dist, prior) , -100, 100)#F.kl_div(pi, prior)
        #print("HELLO??", pi_mean[0].detach().cpu().numpy(), state_batch[0].cpu().numpy(), kl[0].detach().cpu().numpy())
        #print("Tf??", self.prior(state_batch[0])[0], prior.loc[0].detach().cpu().numpy())
        if torch.isnan(kl.mean()):
            print("BAD")
            assert False, "end"
        policy_loss = ((self.alpha * kl) - min_qf_pi).mean()
        self.critic_optim.zero_grad()
        qf1_loss.backward()
        self.critic_optim.step()

        self.critic_optim.zero_grad()
        qf2_loss.backward()
        self.critic_optim.step()

        #self.critic_lr_scheduler.step()
        if torch.isnan(policy_loss.mean()):
            return 0, 0, 0, 0, 0
        self.policy_optim.zero_grad()
        policy_loss.backward()
        self.policy_optim.step()

        #self.actor_lr_scheduler.step()
        self.logger.log('train_actor/loss', policy_loss, step)
        self.logger.log('train_actor/entropy', kl.mean(), step)

        self.logger.log('train_critic/loss', qf1_loss + qf2_loss, step)
        if self.automatic_entropy_tuning:
            # just lazily repurposing target entropy as target divergence rn
            #alpha_loss = -(self.log_alpha * (log_pi + self.target_entropy).detach()).mean()
            alpha_loss = self.log_alpha * (self.target_entropy - kl).detach().mean() # not sure if detach is right here
            self.alpha_optim.zero_grad()
            alpha_loss.backward()
            self.alpha_optim.step()
           # self.alpha_lr_scheduler.step()

            self.alpha = self.log_alpha.exp()
            alpha_tlogs = self.alpha.clone() # For TensorboardX logs
            self.logger.log('train_alpha/loss', alpha_loss, step)
            self.logger.log('train_alpha/value', self.alpha, step)
        else:
            alpha_loss = torch.tensor(0.).to(self.device)
            alpha_tlogs = torch.tensor(self.alpha) # For TensorboardX logs


        if updates % self.target_update_interval == 0:
            soft_update(self.critic_target, self.critic, self.tau)

        return qf1_loss.item(), qf2_loss.item(), policy_loss.item(), alpha_loss.item(), alpha_tlogs.item()

    # Save model parameters
    def save_model(self, env_name, suffix="", actor_path=None, critic_path=None):
        if not os.path.exists('models/'):
            os.makedirs('models/')

        if actor_path is None:
            actor_path = "models/sac_actor_{}_{}".format(env_name, suffix)
        if critic_path is None:
            critic_path = "models/sac_critic_{}_{}".format(env_name, suffix)
        print('Saving models to {} and {}'.format(actor_path, critic_path))
        torch.save(self.policy.state_dict(), actor_path)
        torch.save(self.critic.state_dict(), critic_path)

    # Load model parameters
    def load_model(self, actor_path, critic_path):
        print('Loading models from {} and {}'.format(actor_path, critic_path))
        if actor_path is not None:
            self.policy.load_state_dict(torch.load(actor_path))
        if critic_path is not None:
            self.critic.load_state_dict(torch.load(critic_path))

    def kl_divergence(self, q, p):
        """Here self=q and other=p and we compute KL(q, p)"""
        
        eps = 1e-7
        assert (2 * (p.scale + eps) ** 2).min() != 0, "pscale err"
        kl = (((p.scale + eps).log() - (q.scale + eps).log()) + ((q.scale + eps) ** 2 + (q.loc - p.loc) ** 2) \
               / (2 * (p.scale + eps) ** 2) - 0.5).sum(1)[:,None]
        # kl[kl==float("inf")] = 100
        # kl[kl==float("-inf")] = 100
        if not(p.scale.min() > 0 and q.scale.min() > 0):
            pass
           # print(str(kl.min()) + " " + str(kl.max()) + " scales too small")
        return kl

    # try analytic instead
    def mc_kl_divergence(self, p, q, n_samples=10):
        """Computes monte-carlo estimate of KL divergence. n_samples: how many samples are used for the estimate."""
        samples = [p.rsample() for _ in range(n_samples)]
        ps = torch.stack([p.log_prob(x).sum(1) - q.log_prob(x).sum(1) for x in samples], dim=1)
        return ps.mean(dim=1)[:, None]