"""
Gaussian estimator implementation.
"""

import numpy as np
from .base import GradEstimator

class GaussianEstimator(GradEstimator):
    """Gaussian estimator implementation.
    
    This estimator uses Gaussian distributed noise.
    """
    
    def generate_noise(self, x):
        """Generate Gaussian noise.
        
        Args:
            x (numpy.ndarray): The point at which to estimate the gradient.
            
        Returns:
            numpy.ndarray: The noise vector.
        """
        return np.random.normal(scale=1.0, size=x.shape) 
    
class CentralizedGaussianEstimator(GaussianEstimator):
    def estimate(self, f, x): 
        grad = np.zeros_like(x)
        
        for i in range(self.zoo_batch_size):
            v = self.generate_noise(x)
            noisy_x = x + self.mu * v
            noisy_xx = x - self.mu * v
            diff = f(noisy_x) - f(noisy_xx)
            grad += diff * v
            
        return grad / (2 * self.mu * self.zoo_batch_size) 