"""
Uniform estimator implementation.
"""

import numpy as np
from .base import GradEstimator

class UniformEstimator(GradEstimator):
    """Uniform estimator implementation.
    
    This estimator uses uniformly distributed noise on the unit sphere.
    """
    
    def generate_noise(self, x):
        """Generate uniform noise on the unit sphere.
        
        Args:
            x (numpy.ndarray): The point at which to estimate the gradient.
            
        Returns:
            numpy.ndarray: The noise vector.
        """
        ndim = len(x)
        vec = np.random.randn(ndim)
        vec /= np.linalg.norm(vec)
        return vec * np.sqrt(ndim) 
    
class CentralizedUniformEstimator(UniformEstimator): 
    def estimate(self, f, x): 
        grad = np.zeros_like(x)
        
        for i in range(self.zoo_batch_size):
            v = self.generate_noise(x)
            noisy_x = x + self.mu * v
            noisy_xx = x - self.mu * v
            diff = f(noisy_x) - f(noisy_xx)
            grad += diff * v
            
        return grad / (2 * self.mu * self.zoo_batch_size) 