"""
Factory module for creating estimators.
"""

from unbiased_zoo.estimators import (
    GradEstimator,
    UniformEstimator,
    GaussianEstimator,
    UnbiasedEstimator,
    CoordinateEstimator,
    BernoulliEstimator,
)

def create_estimator(estimator_type, **kwargs):
    """Create an estimator instance.
    
    Args:
        estimator_type (str): The type of estimator to create.
        **kwargs: Additional arguments to pass to the estimator constructor.
        
    Returns:
        GradEstimator: An instance of the specified estimator.
        
    Raises:
        ValueError: If the estimator type is unknown.
    """
    estimator_map = {
        "uniform": UniformEstimator,
        "gaussian": GaussianEstimator,
        "unbiased": UnbiasedEstimator,
        "coordinate": CoordinateEstimator,
        "bernoulli": BernoulliEstimator,
    }
    
    if estimator_type not in estimator_map:
        raise ValueError(f"Unknown estimator type: {estimator_type}")
    
    return estimator_map[estimator_type](**kwargs) 