"""
Base class for black-box functions.
"""

class BlackBoxFunction:
    """Base class for black-box functions.
    
    This class defines the interface for black-box functions that can be used
    with gradient estimators.
    
    Attributes:
        input_dim (int): The dimension of the input space.
    """
    
    def __init__(self, input_dim):
        """Initialize a black-box function.
        
        Args:
            input_dim (int): The dimension of the input space.
        """
        self.input_dim = input_dim

    def _f(self, x):
        """Internal implementation of the function.
        
        Args:
            x (numpy.ndarray): Input vector.
            
        Returns:
            float: Function value at x.
        """
        raise NotImplementedError

    def _grad(self, x):
        """Internal implementation of the gradient.
        
        Args:
            x (numpy.ndarray): Input vector.
            
        Returns:
            numpy.ndarray: Gradient at x.
        """
        raise NotImplementedError

    def __call__(self, x):
        """Call the function.
        
        Args:
            x (numpy.ndarray): Input vector.
            
        Returns:
            float: Function value at x.
        """
        return self._f(x)
        
    def grad(self, x):
        """Compute the gradient.
        
        Args:
            x (numpy.ndarray): Input vector.
            
        Returns:
            numpy.ndarray: Gradient at x.
        """
        return self._grad(x) 