"""
Exponential function implementation.
"""

import numpy as np
from .base import BlackBoxFunction

class ExponentialFunction(BlackBoxFunction):
    """Exponential function implementation.
    
    This class implements an exponential function of the form f(x) = e^(a^T x),
    where a is a vector of coefficients (set to all 1's by default).
    
    Attributes:
        input_dim (int): The dimension of the input space.
        a (numpy.ndarray): The coefficient vector.
    """
    
    def __init__(self, input_dim=10):
        """Initialize an exponential function.
        
        Args:
            input_dim (int, optional): The dimension of the input space. Defaults to 10.
        """
        super().__init__(input_dim)
        # Initialize coefficient vector with all 1's
        self.a = np.ones(input_dim)

    def set_coefficients(self, a):
        """Set the coefficient vector.
        
        Args:
            a (numpy.ndarray): The coefficient vector.
        """
        self.a = a

    def _f(self, x):
        """Compute the function value.
        
        Args:
            x (numpy.ndarray): Input vector.
            
        Returns:
            float: Function value at x.
        """
        return np.exp(np.dot(self.a, x))

    def _grad(self, x):
        """Compute the gradient.
        
        Args:
            x (numpy.ndarray): Input vector.
            
        Returns:
            numpy.ndarray: Gradient at x.
        """
        return self.a * np.exp(np.dot(self.a, x)) 