import os
import http
import json
import requests
import re
import string
import time
import pdfplumber
import ssl
import aiohttp
import asyncio
import concurrent
from tqdm import tqdm
from io import BytesIO
from bs4 import BeautifulSoup
from fake_useragent import UserAgent
from nltk.tokenize import sent_tokenize
from requests.exceptions import Timeout
from concurrent.futures import ThreadPoolExecutor
from typing import Optional, Tuple, List, Dict, Union

import urllib3
from urllib.parse import quote_plus


ssl._create_default_https_context = ssl._create_unverified_context
urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)


# ----------------------- Custom Headers -----------------------
user_agent = UserAgent()
headers = {
    'User-Agent': user_agent.random,
    'Referer': 'https://www.google.com/',
    'Accept': 'text/html,application/xhtml+xml,application/xml;q=0.9,image/webp,*/*;q=0.8',
    'Accept-Language': 'en-US,en;q=0.5',
    'Connection': 'keep-alive',
    'Upgrade-Insecure-Requests': '1'
}

# Initialize session with connection pooling
session = requests.Session()
session.headers.update(headers)
# Optimize connection pooling
adapter = requests.adapters.HTTPAdapter(
    pool_connections=100,  # Number of connection pools
    pool_maxsize=100,      # Maximum number of connections in pool
    max_retries=3,         # Retry failed requests
    pool_block=False       # Don't block when pool is full
)
session.mount('http://', adapter)
session.mount('https://', adapter)



# --- Async HTTP helpers ---
async def async_jina_web_search(query, api_key, top_k=10, session=None, retries=5):
    url = f"https://s.jina.ai/{quote_plus(query)}"
    headers = {
        "Accept": "application/json",
        "Authorization": f"Bearer {api_key}",
        "X-Respond-With": "no-content",
    }
    for i in range(retries):
        try:
            async with session.get(url, headers=headers, ssl=False) as resp:
                resp.raise_for_status()
                data = await resp.json()
                return [{"title": item["title"], "url": item["url"], "snippet": item["description"]} for item in data["data"][:top_k]]
        except Exception:
            if i == retries - 1:
                raise
            await asyncio.sleep(0.1)

async def async_fetch_page_content(
    urls: List[str],
    use_jina: bool = False,
    jina_api_key: Optional[str] = None,
    snippets: Optional[Dict[str, str]] = None,
    max_workers: int = 8,
) -> Dict[str, Union[dict, str]]:
    
    loop = asyncio.get_running_loop()
    snippets = snippets or {}
    results: Dict[str, Union[str, dict]] = {}

    def _job(url):
        while True:
            try:
                return url, extract_text_from_url(
                    url,
                    use_jina=use_jina,
                    jina_api_key=jina_api_key,
                    snippet=snippets.get(url),
                )
            except Exception as e:
                time.sleep(3)
                print(f"Error fetching {url}: {e}")
                continue
            

    with ThreadPoolExecutor(max_workers=max_workers) as pool:
        tasks = [loop.run_in_executor(pool, _job, url) for url in urls]
        for coro in asyncio.as_completed(tasks):
            url, data = await coro
            results[url] = data
            await asyncio.sleep(0.2)

    return results

# --- New: Async fetch content using aiohttp ---
async def async_fetch_content(
    urls: list,
    session: aiohttp.ClientSession,
    snippets: dict = None,
    max_length: int = 8000,
    timeout: int = 20
) -> dict:
    """
    This function is used to fetch the content of the URLs.

    Args:
        urls (list): A list of URLs to fetch
        session (aiohttp.ClientSession): aiohttp session
        snippets (dict): {url: snippet} 형태의 dict (선택)
        max_length (int): Maximum length of the text
        timeout (int): Request timeout (seconds)

    Returns:
        dict: {url: content or error message}
    """
    results = {}

    async def fetch(url):
        try:
            async with session.get(url, timeout=timeout, ssl=False) as resp:
                resp.raise_for_status()
                text = await resp.text()
                soup = BeautifulSoup(text, "lxml")
                page_text = soup.get_text(separator=' ', strip=True)
                snippet = snippets.get(url) if snippets else None
                if snippet:
                    idx = page_text.find(snippet)
                    if idx != -1:
                        start = max(0, idx - 500)
                        end = min(len(page_text), idx + len(snippet) + 500)
                        content = page_text[start:end]
                    else:
                        content = page_text[:max_length]
                else:
                    content = page_text[:max_length]
                results[url] = content
        except Exception as e:
            results[url] = f"Error fetching {url}: {e}"

    await asyncio.gather(*(fetch(url) for url in urls))
    return results


def remove_punctuation(text: str) -> str:
    """Remove punctuation from the text."""
    return text.translate(str.maketrans("", "", string.punctuation))

def f1_score(true_set: set, pred_set: set) -> float:
    """Calculate the F1 score between two sets of words."""
    intersection = len(true_set.intersection(pred_set))
    if not intersection:
        return 0.0
    precision = intersection / float(len(pred_set))
    recall = intersection / float(len(true_set))
    return 2 * (precision * recall) / (precision + recall)

def extract_snippet_with_context(full_text: str, snippet: str, context_chars: int = 2500) -> Tuple[bool, str]:
    """
    Extract the sentence that best matches the snippet and its context from the full text.

    Args:
        full_text (str): The full text extracted from the webpage.
        snippet (str): The snippet to match.
        context_chars (int): Number of characters to include before and after the snippet.

    Returns:
        Tuple[bool, str]: The first element indicates whether extraction was successful, the second element is the extracted context.
    """
    try:
        full_text = full_text[:50000]

        snippet = snippet.lower()
        snippet = remove_punctuation(snippet)
        snippet_words = set(snippet.split())

        best_sentence = None
        best_f1 = 0.2

        # sentences = re.split(r'(?<=[.!?]) +', full_text)  # Split sentences using regex, supporting ., !, ? endings
        sentences = sent_tokenize(full_text)  # Split sentences using nltk's sent_tokenize

        for sentence in sentences:
            key_sentence = sentence.lower()
            key_sentence = remove_punctuation(key_sentence)
            sentence_words = set(key_sentence.split())
            f1 = f1_score(snippet_words, sentence_words)
            if f1 > best_f1:
                best_f1 = f1
                best_sentence = sentence

        if best_sentence:
            para_start = full_text.find(best_sentence)
            para_end = para_start + len(best_sentence)
            start_index = max(0, para_start - context_chars)
            end_index = min(len(full_text), para_end + context_chars)
            context = full_text[start_index:end_index]
            return True, context
        else:
            # If no matching sentence is found, return the first context_chars*2 characters of the full text
            return False, full_text[:context_chars * 2]
    except Exception as e:
        return False, f"Failed to extract snippet context due to {str(e)}"

def extract_text_from_url(url, use_jina=False, jina_api_key=None, snippet: Optional[str] = None):
    """
    Extract text from a URL. If a snippet is provided, extract the context related to it.

    Args:
        url (str): URL of a webpage or PDF.
        use_jina (bool): Whether to use Jina for extraction.
        snippet (Optional[str]): The snippet to search for.

    Returns:
        str: Extracted text or context.
    """
    try:
        if use_jina:
            jina_headers = {
                'Authorization': f'Bearer {jina_api_key}',
                'X-Return-Format': 'markdown',
                # 'X-With-Links-Summary': 'true'
            }
            response = requests.get(f'https://r.jina.ai/{url}', headers=jina_headers, verify=False, timeout=10).text # verify=False
            # Remove URLs
            pattern = r"\(https?:.*?\)|\[https?:.*?\]"
            text = re.sub(pattern, "", response).replace('---','-').replace('===','=').replace('   ',' ').replace('   ',' ')
        else:
            response = session.get(url, timeout=10, verify=False)  # Set timeout to 10 seconds
            response.raise_for_status()  # Raise HTTPError if the request failed
            # Determine the content type
            content_type = response.headers.get('Content-Type', '')
            if 'pdf' in content_type:
                # If it's a PDF file, extract PDF text
                return extract_pdf_text(url)
            # Try using lxml parser, fallback to html.parser if unavailable
            try:
                soup = BeautifulSoup(response.text, 'lxml')
            except Exception:
                print("lxml parser not found or failed, falling back to html.parser")
                soup = BeautifulSoup(response.text, 'html.parser')
            text = soup.get_text(separator=' ', strip=True)

        if snippet:
            success, context = extract_snippet_with_context(text, snippet)
            if success:
                return context
            else:
                return text
        else:
            # If no snippet is provided, return directly
            return text[:1024]
    except requests.exceptions.HTTPError as http_err:
        return f"HTTP error occurred: {http_err}"
    except requests.exceptions.ConnectionError as e:
        return "Error: Connection error occurred"
    except requests.exceptions.Timeout:
        return "Error: Request timed out after 10 seconds"
    except Exception as e:
        return f"Unexpected error: {str(e)}"

def fetch_page_content(urls, max_workers=32, use_jina=False, jina_api_key=None, snippets: Optional[dict] = None):
    """
    Concurrently fetch content from multiple URLs with optimized performance.

    Args:
        urls (list): List of URLs to scrape.
        max_workers (int): Maximum number of concurrent threads.
        use_jina (bool): Whether to use Jina for extraction.
        snippets (Optional[dict]): A dictionary mapping URLs to their respective snippets.

    Returns:
        dict: A dictionary mapping URLs to the extracted content or context.
    """
    results = {}
    with ThreadPoolExecutor(max_workers=max_workers) as executor:
        # Use tqdm to display a progress bar
        futures = {
            executor.submit(extract_text_from_url, url, use_jina, jina_api_key, snippets.get(url) if snippets else None): url
            for url in urls
        }
        for future in tqdm(concurrent.futures.as_completed(futures), desc="Fetching URLs", total=len(urls)):
            url = futures[future]
            try:
                data = future.result()
                results[url] = data
            except Exception as exc:
                results[url] = f"Error fetching {url}: {exc}"
            # Remove rate limiting delay for better performance
            # time.sleep(0.2)  # Simple rate limiting
    return results



def jina_web_search(query, subscription_key, top_k, depth=0):
    search_url = f"https://s.jina.ai/{query.replace(' ', '+')}"
    search_headers = {
        'Accept': 'application/json',
        'Authorization': f'Bearer {subscription_key}',
        'X-Respond-With': 'no-content'
    }
    results = []

    try:
        response = requests.get(search_url, headers=search_headers, verify=False)
        response = json.loads(response.text) 
        for e in response["data"]:
            results.append({
                "title": e['title'],
                "url": e['url'],
                "snippet": e['description']
            })
    except Exception as e:
        print(f"Jina search API error: {e}")
        if depth < 128:
            time.sleep(1)
            return jina_web_search(query, subscription_key, top_k, depth+1) 
        else:
            print(f"Jina search API error: {e}")
            return []
    return results



def azure_bing_search(query, subscription_key, mkt, top_k, depth=0):
    params = {'q': query, 'mkt': mkt, 'count': top_k}
    headers = {'Ocp-Apim-Subscription-Key': subscription_key}

    results = []

    try:
        response = requests.get("https://api.bing.microsoft.com/v7.0/search", headers=headers, params=params)
        json_response = response.json()
        for e in json_response['webPages']['value']:
            results.append({
                "title": e['name'],
                "link": e['url'],
                "snippet": e['description']
            })
    except Exception as e:
        print(f"Bing search API error: {e}")
        if depth < 8:
            time.sleep(0.1)
            return azure_bing_search(query, subscription_key, mkt, top_k, depth+1)
    return results


def serper_google_search(
        query, 
        serper_api_key,
        top_k,
        region="us",
        lang="en",
        depth=0
    ):
    try:
        conn = http.client.HTTPSConnection("google.serper.dev")
        payload = json.dumps({
                "q": query,
                "num": top_k,
                "gl": region,
                "hl": lang,
            })
        headers = {
            'X-API-KEY': serper_api_key,
            'Content-Type': 'application/json'
        }
        conn.request("POST", "/search", payload, headers)
        res = conn.getresponse()
        data = json.loads(res.read().decode("utf-8"))
        
        print(data)
        if not data:
            raise Exception("The google search API is temporarily unavailable, please try again later.")

        if "organic" not in data:
            raise Exception(f"No results found for query: '{query}'. Use a less specific query.")
        else:
            results = data["organic"]
            print("search success")
            return results
    except Exception as e:
        print(f"Serper search API error: {e}")
        if depth < 8:
            time.sleep(0.1)
            return serper_google_search(query, serper_api_key, top_k, region, lang, depth=depth+1)
    print("search failed")
    return []


def extract_pdf_text(url):
    """
    Extract text from a PDF.

    Args:
        url (str): URL of the PDF file.

    Returns:
        str: Extracted text content or error message.
    """
    try:
        response = session.get(url, timeout=20)  # Set timeout to 20 seconds
        if response.status_code != 200:
            return f"Error: Unable to retrieve the PDF (status code {response.status_code})"
        
        # Open the PDF file using pdfplumber
        with pdfplumber.open(BytesIO(response.content)) as pdf:
            full_text = ""
            for page in pdf.pages:
                text = page.extract_text()
                if text:
                    full_text += text
        
        # Limit the text length
        cleaned_text = ' '.join(full_text.split()[:600])
        return cleaned_text
    except requests.exceptions.Timeout:
        return "Error: Request timed out after 20 seconds"
    except Exception as e:
        return f"Error: {str(e)}"

def extract_relevant_info(search_results):
    """
    Extract relevant information from Bing search results.

    Args:
        search_results (dict): JSON response from the Bing Web Search API.

    Returns:
        list: A list of dictionaries containing the extracted information.
    """
    useful_info = []
    
    if 'webPages' in search_results and 'value' in search_results['webPages']:
        for id, result in enumerate(search_results['webPages']['value']):
            info = {
                'id': id + 1,  # Increment id for easier subsequent operations
                'title': result.get('name', ''),
                'url': result.get('url', ''),
                'site_name': result.get('siteName', ''),
                'date': result.get('datePublished', '').split('T')[0],
                'snippet': result.get('snippet', ''),  # Remove HTML tags
                # Add context content to the information
                'context': ''  # Reserved field to be filled later
            }
            useful_info.append(info)
    
    return useful_info


# ------------------------------------------------------------

if __name__ == "__main__":
    # Example usage
    # Define the query to search
    query = "Structure of dimethyl fumarate"
    
    # Subscription key and endpoint for Bing Search API
    BING_SUBSCRIPTION_KEY = "YOUR_BING_SUBSCRIPTION_KEY"
    if not BING_SUBSCRIPTION_KEY:
        raise ValueError("Please set the BING_SEARCH_V7_SUBSCRIPTION_KEY environment variable.")
    
    bing_endpoint = "https://api.bing.microsoft.com/v7.0/search"
    
    # Perform the search
    print("Performing Bing Web Search...")
    search_results = bing_web_search(query, BING_SUBSCRIPTION_KEY, bing_endpoint)
    
    print("Extracting relevant information from search results...")
    extracted_info = extract_relevant_info(search_results)

    print("Fetching and extracting context for each snippet...")
    for info in tqdm(extracted_info, desc="Processing Snippets"):
        full_text = extract_text_from_url(info['url'], use_jina=True)  # Get full webpage text
        if full_text and not full_text.startswith("Error"):
            success, context = extract_snippet_with_context(full_text, info['snippet'])
            if success:
                info['context'] = context
            else:
                info['context'] = f"Could not extract context. Returning first 8000 chars: {full_text[:8000]}"
        else:
            info['context'] = f"Failed to fetch full text: {full_text}"

    # print("Your Search Query:", query)
    # print("Final extracted information with context:")
    # print(json.dumps(extracted_info, indent=2, ensure_ascii=False))
