from ray import tune, train
import ray
import os
import torch
from torch.utils.data import DataLoader, TensorDataset
import random
import numpy as np
seed = 30
from models.pendulum import PendulumNet
torch.manual_seed(seed)
random.seed(seed)
np.random.seed(seed)

b = 0.5

import argparse
parser = argparse.ArgumentParser()

parser.add_argument('--mode', default='Derivative', type=str, help='Mode: -1 for PINN learning, 0 for derivative learning, 1 for output learning')
parser.add_argument('--device', default='cuda:0', type=str, help='Device to run the code')
parser.add_argument('--name', default='true', type=str, help='Experiment name')
args = parser.parse_args()
mode = args.mode
device_in = args.device
name = args.name
if b == 0:
    pendulum_type = 'pendulum_conservative'
else:
    pendulum_type = 'pendulum_damped'

if name == 'true':
    prefix = 'true'
elif name == 'extrapolate':
    prefix = 'true_extrapolate'
elif name == 'interpolate':
    prefix = 'true_interpolate'
elif name == 'adapt':
    prefix = 'true_adapt'
elif name == 'emp':
    prefix = 'emp'
else:
    raise ValueError(f'name value is not in the options')

exp_name = pendulum_type+'_'+name
print('exp_name:', exp_name)

def train_model(config):
    #ray.train.torch.enable_reproducibility(seed=seed)
    torch.manual_seed(seed)
    random.seed(seed)
    np.random.seed(seed)
    os.environ["CUDA_VISIBLE_DEVICES"] = "0,1,2,3,4,5,6,7"
    EXP_PATH = f'{os.path.abspath('.')}/{pendulum_type}'
    epochs = 200
    print(mode)
    
    batch_size = 32
    init_weight = config['init_weight']
    if mode != 'PINN':
        sys_weight = config['sys_weight']
    else:
        sys_weight = 0.
    if mode == 'PINN':
        pde_weight = config['pde_weight']
    else:
        pde_weight = 0.
    lr_init = config['lr_init']
    
    # Load the data
    train_dataset = torch.load(os.path.join(EXP_PATH, f'{prefix}_dataset_train.pth'))
    val_dataset = torch.load(os.path.join(EXP_PATH, f'{prefix}_dataset_val.pth'))
    
    if name in ['adapt', 'interpolate']:
        bc_dataset = torch.load(os.path.join(EXP_PATH, f'{prefix}_bc_train.pth'))
    else:
        bc_dataset = None

    # Generate the dataloaders
    train_dataloader = DataLoader(train_dataset, batch_size, shuffle=True, num_workers=2, generator=torch.Generator().manual_seed(seed))
    val_dataloader = DataLoader(val_dataset, batch_size, shuffle=True, num_workers=2, generator=torch.Generator().manual_seed(seed))

    device = "cpu"
    if torch.cuda.is_available():
        device = device_in
    
    activation = torch.nn.Tanh()
    model = PendulumNet(
        init_weight=init_weight,
        sys_weight=sys_weight,
        pde_weight=pde_weight,
        hidden_units=[20 for _ in range(4)],
        lr_init=lr_init,
        device=device,
        activation=activation,    
        last_activation=False,
    ).to(device)
    
    
    
    # Training mode for the network
    model.train()
    
    for epoch in range(epochs):
        step_prefix = epoch*len(train_dataloader)
        print(f'Epoch: {epoch}, step_prefix: {step_prefix}')
        for step, train_data in enumerate(train_dataloader):
            # Load batches from dataloaders
            x_train = train_data[0].to(device).float()
            
            y_train = train_data[1].to(device).float()
            u_train = train_data[2].to(device).float()
            
            if bc_dataset is not None:
                x_bc = bc_dataset[:][0].to(device).float()
                y_bc = bc_dataset[:][1].to(device).float()
            else:
                x_bc = None
                y_bc = None
            
            # Call zero grad on optimizer
            model.opt.zero_grad()
            
            loss = model.loss_fn(mode=mode,
                x=x_train, y=y_train, u=u_train, x_bc=x_bc, y_bc=y_bc
            )
            # Backward the loss, calculate gradients
            loss.backward()
            # Optimizer step
            model.opt.step()
            # Update the learning rate scheduling
        
        out_losses = []
        der_losses = []
        pde_losses = []
        init_losses = []
        tot_losses = []
        init_pde_losses = []
        for step, val_data in enumerate(val_dataloader):
            x_val = val_data[0].to(device).float()
            y_val = val_data[1].to(device).float()
            u_val = val_data[2].to(device).float()
        
            # Printing
            with torch.no_grad():
                #step_val, out_loss_train, der_loss_train, pde_loss_train, init_loss_train, tot_loss_train, init_pde_loss_train = model.eval_losses(
                #    step=step_prefix+step, mode=mode,
                #    x=x_train, y=y_train, u=u_train, x_bc=None, y_bc=None
                #)
                step_val, out_loss_val, der_loss_val, pde_loss_val, init_loss_val, tot_loss_val, init_pde_loss_val = model.eval_losses(
                    step=step_prefix+step, mode=mode,
                    x=x_val, y=y_val, u=u_val, x_bc=x_bc, y_bc=y_bc
                )
            out_losses.append(out_loss_val.item())
            der_losses.append(der_loss_val.item())
            pde_losses.append(pde_loss_val.item()) 
            init_losses.append(init_loss_val.item())
            tot_losses.append(tot_loss_val.item())
            init_pde_losses.append(init_pde_loss_val.item())
            
        out_loss_val = np.mean(out_losses)
        der_loss_val = np.mean(der_losses)
        pde_loss_val = np.mean(pde_losses)
        init_loss_val = np.mean(init_losses)
        tot_loss_val = np.mean(tot_losses)
        init_pde_loss_val = np.mean(init_pde_losses)
            
        train.report({
            'step': step_prefix+step,
            'loss': tot_loss_val.item(),
            'out_loss': out_loss_val.item(),
            'der_loss': der_loss_val.item(),
            'pde_loss': pde_loss_val.item(),
            'init_loss': init_loss_val.item(),
            'init_pde_loss': init_pde_loss_val.item(),
        })
                
                
                
                             
param_space = {
    "init_weight": tune.loguniform(1e-3, 1e2),
    "lr_init": tune.choice([1e-3, 5e-4, 1e-4]),
}

if mode != 'PINN':
    param_space['sys_weight'] = tune.loguniform(1e-3, 1e2)
if mode == 'PINN':
    param_space['pde_weight'] = tune.loguniform(1e-3, 1e2)

from ray.tune.search.hyperopt import HyperOptSearch
from ray.tune.schedulers import ASHAScheduler

algo = HyperOptSearch()     
scheduler = ASHAScheduler(max_t=200, grace_period=50)
           
trainable_with_resources = tune.with_resources(train_model, {'cpu':4, 'gpu':0.25})
           
tuner = tune.Tuner(
    trainable_with_resources,
    tune_config=tune.TuneConfig(metric="out_loss", mode="min", search_alg=algo, scheduler=scheduler, num_samples=100, max_concurrent_trials=4,),
    run_config=train.RunConfig(storage_path=f'{os.path.abspath('.')}/results_new', name=f'{exp_name}{mode}_tris', log_to_file=True),
    param_space=param_space,
)

results = tuner.fit()

best_result = results.get_best_result()  # Get best result object
best_config = best_result.config  # Get best trial's hyperparameters
best_logdir = best_result.path  # Get best trial's result directory
best_checkpoint = best_result.checkpoint  # Get best trial's best checkpoint
best_metrics = best_result.metrics  # Get best trial's last results
best_result_df = best_result.metrics_dataframe  # Get best result as pandas dataframe

print(best_result)
print(best_config)
print(best_logdir)
print(best_checkpoint)
print(best_metrics)
print(best_result_df)
