from ray import tune, train
import ray
import os
import torch
from torch.utils.data import DataLoader, TensorDataset
import random
import numpy as np

seed = 30
from torch.func import vmap, jacrev
from itertools import cycle

from models.ns import NSRestrictedNet
torch.manual_seed(seed)
random.seed(seed)
np.random.seed(seed)

import argparse
parser = argparse.ArgumentParser()

parser.add_argument('--mode', default='Derivative', type=str, help='Mode: -1 for PINN learning, 0 for derivative learning, 1 for output learning')
parser.add_argument('--device', default='cuda:0', type=str, help='Device to run the code')

args = parser.parse_args()
mode = args.mode
device_in = args.device


def train_model(config):
    os.environ["CUDA_VISIBLE_DEVICES"] = "0,1,2,3,4,5,6,7,8"
    EXP_PATH = f'{os.path.abspath('.')}/NS_true_restricted'
    epochs = 500
    batch_size = 512
    lr_init = config['lr_init']
    #batch_size = config['batch_size']
    init_weight = config['init_weight']
    bc_weight = config['bc_weight']
    if mode == 'PINN':
        mom_weight = config['mom_weight']
        div_weight = config['div_weight']
    else:
        mom_weight = 0.
        div_weight = 0.
    if mode != 'PINN':
        sys_weight = config['sys_weight']
    else:
        sys_weight = 0
    
    # Load the data
    init_dataset = torch.load(os.path.join(EXP_PATH, f'init_data.pth'))
    bc_dataset = torch.load(os.path.join(EXP_PATH, f'bc_data.pth'))
    train_dataset = torch.load(os.path.join(EXP_PATH, f'pde_data.pth'))

    # Generate the dataloaders
    train_dataloader = DataLoader(train_dataset, batch_size, shuffle=True, num_workers=6)
    bc_dataloader = DataLoader(bc_dataset, batch_size, shuffle=True, num_workers=6)
    init_dataloader = DataLoader(init_dataset, batch_size, shuffle=True, num_workers=6)

    device = "cpu"
    if torch.cuda.is_available():
        device = device_in
    
    activation = torch.nn.Tanh()
    
    model = NSRestrictedNet(
        init_weight=init_weight,
        mom_weight=mom_weight,
        sys_weight=sys_weight,
        div_weight=div_weight,
        bc_weight=bc_weight,
        hidden_units=[128 for i in range(8)],
        lr_init=lr_init,
        activation=activation,
        device=device,
        last_activation=False,
    ).to(device)
    
    
    
    # Training mode for the network
    model.train()
    
    for epoch in range(epochs):
        model.train()
        step_prefix = epoch*len(train_dataloader)
        print(f'Epoch: {epoch}, step_prefix: {step_prefix}')
        for step, (pde_data, init_data, bc_data) in enumerate(zip(train_dataloader, cycle(init_dataloader), cycle(bc_dataloader))):
            # Load batches from dataloaders
            x_pde = pde_data[0].to(device).float().requires_grad_(True)
            y_pde = pde_data[1].to(device).float()
            D_pde = pde_data[2].reshape((-1,3,3)).to(device).float()
                     
            x_init = init_data[0].to(device).float()
            y_init = init_data[1].to(device).float()
            
            x_bc = bc_data[0].to(device).float()
            y_bc = bc_data[1].to(device).float()
            
            
            
            # Call zero grad on optimizer
            model.opt.zero_grad()
            
            loss = model.loss_fn(mode=mode, x_pde=x_pde, y_pde=y_pde, D_pde=D_pde, x_init=x_init, y_init=y_init, x_bc=x_bc, y_bc=y_bc)
            # Backward the loss, calculate gradients
            loss.backward()
            # Optimizer step
            model.opt.step()
            # Update the learning rate scheduling
        
        out_losses = []
        der_losses = []
        mom_losses = []
        div_losses = []
        init_losses = []
        bc_losses = []
        tot_losses = []
        for step, (pde_data, init_data, bc_data) in enumerate(zip(train_dataloader, cycle(init_dataloader), cycle(bc_dataloader))):
            # Load batches from dataloaders
            x_pde = pde_data[0].to(device).float().requires_grad_(True)
            y_pde = pde_data[1].to(device).float()
            D_pde = pde_data[2].reshape((-1,3,3)).to(device).float()
                     
            x_init = init_data[0].to(device).float()
            y_init = init_data[1].to(device).float()
            
            x_bc = bc_data[0].to(device).float()
            y_bc = bc_data[1].to(device).float()
            # Printing
            with torch.no_grad():
                #step_val, out_loss_train, der_loss_train, pde_loss_train, init_loss_train, tot_loss_train, init_pde_loss_train = model.eval_losses(
                #    step=step_prefix+step, mode=mode,
                #    x=x_train, y=y_train, u=u_train, x_bc=None, y_bc=None
                #)
                step_val, mom_loss_val, div_loss_val, out_loss_val, der_loss_val, init_loss_val, bc_loss_val, tot_loss_val = model.print_losses(
                    step=step_prefix+step, mode=mode, x_pde=x_pde, y_pde=y_pde, D_pde=D_pde, x_init=x_init, y_init=y_init, x_bc=x_bc, y_bc=y_bc, print_to_screen=False
                )
            out_losses.append(out_loss_val.item())
            der_losses.append(der_loss_val.item())
            mom_losses.append(mom_loss_val.item())
            div_losses.append(div_loss_val.item())
            init_losses.append(init_loss_val.item())
            bc_losses.append(bc_loss_val.item())
            tot_losses.append(tot_loss_val.item())
            
        out_loss_val = np.mean(out_losses)
        der_loss_val = np.mean(der_losses)
        mom_loss_val = np.mean(mom_losses)
        div_loss_val = np.mean(div_losses)
        init_loss_val = np.mean(init_losses)
        bc_loss_val = np.mean(bc_losses)
        tot_loss_val = np.mean(tot_losses)
            
        train.report({
            'step': step_prefix+step,
            'loss': tot_loss_val.item(),
            'out_loss': out_loss_val.item(),
            'der_loss': der_loss_val.item(),
            'mom_loss': mom_loss_val.item(),
            'div_loss': div_loss_val.item(),
            'init_loss': init_loss_val.item(),
            'bc_loss': bc_loss_val.item(),
            'tot_loss': tot_loss_val.item(),
        })
                
                
                
                             
param_space = {
    #"batch_size": tune.choice([64, 128, 256, 512]),
    "init_weight": tune.loguniform(1e-3, 1e2),
    "bc_weight": tune.loguniform(1e-3, 1e2),
    "lr_init": tune.choice([1e-3, 5e-4, 1e-4]),
}

if mode == 'PINN':
    param_space['mom_weight'] = tune.loguniform(1e-3, 1e2)
    param_space['div_weight'] = tune.loguniform(1e-3, 1e2)
if mode != 'PINN':
    param_space['sys_weight'] = tune.loguniform(1e-3, 1e2)

from ray.tune.search.hyperopt import HyperOptSearch
from ray.tune.schedulers import ASHAScheduler

algo = HyperOptSearch()     
scheduler = ASHAScheduler(max_t=500, grace_period=100)
           
trainable_with_resources = tune.with_resources(train_model, {'cpu':18, 'gpu':0.25})
           
tuner = tune.Tuner(
    trainable_with_resources,
    tune_config=tune.TuneConfig(metric="out_loss", mode="min", search_alg=algo, scheduler=scheduler, num_samples=50, max_concurrent_trials=4,),
    run_config=train.RunConfig(storage_path=f'{os.path.abspath('.')}/results_new', name=f'ns_true_restricted{mode}', log_to_file=True),
    param_space=param_space,
)

results = tuner.fit()

best_result = results.get_best_result()  # Get best result object
best_config = best_result.config  # Get best trial's hyperparameters
best_logdir = best_result.path  # Get best trial's result directory
best_checkpoint = best_result.checkpoint  # Get best trial's best checkpoint
best_metrics = best_result.metrics  # Get best trial's last results
best_result_df = best_result.metrics_dataframe  # Get best result as pandas dataframe

print(best_result)
print(best_config)
print(best_logdir)
print(best_checkpoint)
print(best_metrics)
print(best_result_df)