# --- graph_utils.py ---
import numpy as np

def build_adj_matrix(m: int, graph: str):
    if graph == 'ring':
        V = np.identity(m)
        for i in range(m-1):
            V[i,i+1] = 1
            V[i,i-1] = 1
        V[m-1,0] = 1
        V[m-1,m-2] = 1
    elif graph == 'random':
        V = np.zeros((m, m))
        perm = np.random.permutation(m)
        for i in range(m-1):
            u, v = perm[i], perm[i+1]
            V[u, v] = V[v, u] = 1
        pv = 0.3
        for i in range(m):
            for j in range(i+1, m):
                if V[i, j] == 0 and np.random.rand() < pv:
                    V[i, j] = V[j, i] = 1
    elif graph == 'connected':
        V = np.ones((m,m))
    else:
        raise ValueError(f'Unknown graph type: {graph}')
    return V

def build_weights(V: np.ndarray):
    m = V.shape[0]
    W = np.zeros_like(V, dtype=float)
    N = np.sum(V, axis=1) - 1
    for i in range(m):
        for j in range(m):
            if V[i,j] == 1 and i != j:
                W[i,j] = 1/(1 + max(N[i],N[j]))
    for i in range(m):
        W[i,i] = 1 - np.sum(W[:,i])
    return W

def build_laplacian(V: np.ndarray):
    D = np.diag(np.sum(V, axis=1))
    return D - V
