# CBM model for MNIST
import torch
import torch.nn as nn
from utils.args import *
from utils.conf import get_device
from utils.losses import *
from utils.dpl_loss import CUB_DPL
from models.utils.cbm_module import CBMModule


def get_parser() -> ArgumentParser:
    """Returns the parser

    Returns:
        argparse: argument parser
    """
    parser = ArgumentParser(description="Learning via" "Concept Extractor .")
    add_management_args(parser)
    add_experiment_args(parser)
    return parser


class CUBCBM(CBMModule):
    NAME = "cubcbm"

    def __init__(
        self,
        encoder,
        n_images=1,
        c_split=(),
        args=None,
        model_dict=None,
        n_facts=312,
        nr_classes=200,
    ):
        """Initialize method

        Args:
            self: instance
            encoder (nn.Module): encoder
            n_images (int, default=2): number of images
            c_split: concept splits
            args: command line arguments
            model_dict (default=None): model dictionary
            n_facts (int, default=20): number of concepts
            nr_classes (int, nr_classes): number of classes

        Returns:
            None: This function does not return a value.
        """
        super(CUBCBM, self).__init__(
            encoder=encoder,
            model_dict=model_dict,
            n_facts=n_facts,
            nr_classes=nr_classes,
        )

        # how many images and explicit split of concepts
        self.n_images = n_images
        self.c_split = c_split

        # opt and device
        self.opt = None
        self.device = get_device()
        self.classifier = nn.Sequential(
            nn.Linear(self.n_facts, self.nr_classes, bias=False),
            nn.Softmax(dim=1),
        )

        torch.nn.init.normal_(self.classifier[0].weight, mean=0.0, std=10.0)

    def forward(self, x):
        """Forward method

        Args:
            self: instance
            x (torch.tensor): input vector

        Returns:
            out_dict: output dict
        """
        # Image encoding
        cs = self.encoder(x)[0]
        # normalize concept preditions
        pCs = torch.sigmoid(cs)
        # get the result of the inference
        py = self.cmb_inference(pCs)

        return {"CS": cs, "YS": py, "pCS": pCs, "KNOWLEDGE": self.classifier}


    def cmb_inference(self, pCs, query=None):
        """Performs inference inference

        Args:
            self: instance
            cs: concepts logits
            query (default=None): query

        Returns:
            query_prob: query probability
        """
        return self.classifier(pCs.squeeze(1))


    def normalize_concepts(self, z, split=2):
        eps = 1e-5
        prob = F.softmax(z, dim=-1) + eps
        prob /= prob.sum(dim=-1, keepdim=True)
        return prob

    @staticmethod
    def get_loss(args):
        """Loss function for the architecture

        Args:
            args: command line arguments

        Returns:
            loss: loss function

        Raises:
            err: NotImplementedError if the loss function is not available
        """
        if args.dataset in [
            "cub",
        ]:
            return CUB_DPL(CUB_Cumulative)
        else:
            return NotImplementedError("Wrong dataset choice")

    def start_optim(self, args):
        """Initialize optimizer

        Args:
            self: instance
            args: command line arguments

        Returns:
            None: This function does not return a value.
        """
        self.opt = torch.optim.Adam(
            self.parameters(), args.lr, weight_decay=args.weight_decay
        )
