"""
    Standard Libraries
"""
from abc import ABC, abstractmethod

"""
    3rd Party Libraries
"""
import numpy as np


class Distribution(ABC):
    """
        Abstract base class for all distributions in the Assessment.

        This class defines the interface that all distribution implementations must follow, 
        ensuring consistency across the library.
    """
    def __init__(self, dim, name=None, seed=None):
        """
            Initialize a distribution.

            Args:
                dim (int): Dimensionality of the distribution
                name (str, optional): Name identifier for the distribution
                seed (int, optional): Random seed for reproducibility
        """
        self.dim = dim
        self.name = name if name is not None else self.__class__.__name__
        self.seed = seed

        if seed is not None:
            np.random.seed(seed)


    def __str__(self):
        """
            String representation of the distribution.
        """
        return f"{self.name} (dim={self.dim})"
    

    @abstractmethod
    def sample(self, n_samples):
        """
            Generate samples from the distribution.
        
            Args:
                n_samples (int): Number of samples to generate
                
            Returns:
                np.ndarray: Generated samples with shape (n_samples, dim)
        """
        pass


    def log_prob(self, x):
        """
            Compute log probability density at the given points.

            Args:
                x (np.ndarray): Points at which to evaluate log probability
                
            Returns:
                np.ndarray: Log probability values
                
            Raises:
                NotImplementedError: If the distribution doesn't support exact log probability
        """
        raise NotImplementedError(
            f"Log probability not implemented for {self.name}"
        )
    

    def score(self, x):
        """
            Compute score function (gradient of log probability) at the given points.

            Args:
                x (np.ndarray): Points at which to evaluate the score
                
            Returns:
                np.ndarray: Score vectors
                
            Raises:
                NotImplementedError: If the distribution doesn't support exact score computation
        """
        raise NotImplementedError(
            f"Score function not implemented for {self.name}"
        )
