"""
    Standard Libraries
"""
from abc import ABC, abstractmethod


class Metric(ABC):
    """
        Abstract base class for all distribution metrics.
        
        This class defines the interface for metrics that evaluate
        the quality of distribution approximations.
    """
    def __init__(self, name=None):
        """
            Initialize a metric.
            
            Args:
                name (str, optional): Name identifier for the metric
        """
        self.name = name if name is not None else self.__class__.__name__


    @abstractmethod
    def compute(self, target_samples, approx_samples, **kwargs):
        """
            Compute the metric between two sets of samples.
            
            Args:
                target_samples (np.ndarray): Samples from the target distribution
                approx_samples (np.ndarray): Samples from the approximation
                **kwargs: Additional arguments specific to the metric
                
            Returns:
                float: Computed metric value
        """
        pass


    def __str__(self):
        """
            String representation of the metric.
        """
        return self.name
