import numpy as np
import warnings
from escort.escort_improvements import ImprovedESCORT, ImprovedSVGD, ImprovedRBFKernel, ImprovedGSWD

class ESCORTNoTemp(ImprovedESCORT):
    """
    ESCORT variant without temporal consistency constraints.
    Maintains the same interface as ImprovedESCORT but disables temporal consistency.
    """
    def __init__(self, n_particles=100, state_dim=2, kernel_bandwidth=0.1, 
                step_size=0.01, lambda_corr=0.1, lambda_temp=0.1, 
                n_projections=10, learning_rate=1e-3, device=None, 
                discrete_actions=True, verbose=False):
        """
        Initialize the ESCORT-NoTemp agent, setting lambda_temp to 0.
        """
        # Call parent constructor with lambda_temp set to 0
        super().__init__(
            n_particles=n_particles,
            state_dim=state_dim,
            kernel_bandwidth=kernel_bandwidth,
            step_size=step_size,
            lambda_corr=lambda_corr,
            lambda_temp=0.0,  # Force lambda_temp to 0 to disable temporal consistency
            n_projections=n_projections,
            learning_rate=learning_rate,
            device=device,
            discrete_actions=discrete_actions,
            verbose=verbose
        )
        
        if verbose:
            print("Initialized ESCORT-NoTemp (temporal consistency disabled)")
    
    def update(self, action, observation, transition_model, observation_model):
        """
        Override update method to ensure temporal consistency is disabled.
        """
        # Store original lambda_temp value
        original_lambda_temp = self.lambda_temp
        
        # Force lambda_temp to 0 during update
        self.lambda_temp = 0.0
        self.svgd.lambda_temp = 0.0
        
        # Call parent update method
        super().update(action, observation, transition_model, observation_model)
        
        # Restore original value (though it should already be 0)
        self.lambda_temp = original_lambda_temp
