# Copyright 2025 The HuggingFace Team. All rights reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging
import os
import sys

import datasets
import torch
import transformers
from datasets import load_dataset
from transformers import set_seed
from transformers.trainer_utils import get_last_checkpoint

from open_r1.configs import GRPOConfig, GRPOScriptArguments
from open_r1.rewards import get_reward_funcs
from open_r1.utils import get_tokenizer
from open_r1.utils.callbacks import get_callbacks
from open_r1.utils.wandb_logging import init_wandb_training

from open_r1.multi_turn_grpo_trainer import MultiTurnGRPOTrainer
from open_r1.envs.augmented_question_env import AugmentedQuestionEnv
from trl import GRPOTrainer, ModelConfig, TrlParser, get_peft_config


logger = logging.getLogger(__name__)


def main(script_args, training_args, model_args):
    # Set seed for reproducibility
    set_seed(training_args.seed)

    ###############
    # Setup logging
    ###############
    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%Y-%m-%d %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )
    log_level = training_args.get_process_log_level()
    logger.setLevel(log_level)
    datasets.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.enable_default_handler()
    transformers.utils.logging.enable_explicit_format()

    # Log on each process a small summary
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f" distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    logger.info(f"Model parameters {model_args}")
    logger.info(f"Script parameters {script_args}")
    logger.info(f"Training parameters {training_args}")

    # Check for last checkpoint
    last_checkpoint = None
    if os.path.isdir(training_args.output_dir):
        last_checkpoint = get_last_checkpoint(training_args.output_dir)
    if last_checkpoint is not None and training_args.resume_from_checkpoint is None:
        logger.info(f"Checkpoint detected, resuming training at {last_checkpoint=}.")

    if "wandb" in training_args.report_to:
        init_wandb_training(training_args)

    # Load the dataset
    if script_args.dataset_remote_or_local == "remote":
        # Load the dataset from the HuggingFace Hub
        dataset = load_dataset(script_args.dataset_name, name=script_args.dataset_config)
    else:
        # Load the dataset from the local path, dataset should be in json format
        try:
            assert script_args.dataset_local_train_path is not None and script_args.dataset_local_test_path is not None
        except AssertionError:
            raise ValueError("dataset_local_train_path and dataset_local_test_path must be provided if dataset_remote_or_local is 'local'")
        dataset = load_dataset("json", data_files={"train": script_args.dataset_local_train_path,
                                                   "test": script_args.dataset_local_test_path})

    ################
    # Load tokenizer
    ################
    tokenizer = get_tokenizer(model_args, training_args)

    # Get reward functions from the registry
    reward_funcs = get_reward_funcs(script_args)

    # # Format into conversation
    # def make_conversation(example, prompt_column: str = script_args.dataset_prompt_column):
    #     prompt = []

    #     if training_args.system_prompt is not None:
    #         prompt.append({"role": "system", "content": training_args.system_prompt})

    #     if prompt_column not in example:
    #         raise ValueError(f"Dataset Question Field Error: {prompt_column} is not supported.")

    #     prompt.append({"role": "user", "content": example[prompt_column]})
    #     return {"prompt": prompt}

    # dataset = dataset.map(make_conversation)

    ##############################
    # Initialize the MultiTurn Env
    ##############################
    message_end_newline_id = None
    if 'qwen' in tokenizer.name_or_path.lower():
        print(f"Qwen2Tokenizer detected, using `<|im_end|>\n` as message end tokens based on the chat template.")
        message_end_id = tokenizer.encode(text='<|im_end|>', add_special_tokens=False)[0]
        message_end_newline_id = 198                    # This is the id for `\n` in Qwen2Tokenizer, used in the chat template
    elif 'gemma' in tokenizer.name_or_path.lower():
        print(f"GemmaTokenizer detected, using `<end_of_turn>\n` as message end tokens based on the chat template.")
        message_end_id = tokenizer.encode(text='<end_of_turn>', add_special_tokens=False)[0]
        message_end_newline_id = 108                    # This is the id for `\n` in GemmaTokenizer, used in the chat template
    else:
        message_end_id = tokenizer.eos_token_id
        message_end_newline_id = None
    
    if 'qwen' in tokenizer.name_or_path.lower():
        support_system_prompt = True
    elif 'gemma-3' in tokenizer.name_or_path.lower():
        support_system_prompt = True
    elif 'llama' in tokenizer.name_or_path.lower():
        support_system_prompt = True
    else:
        support_system_prompt = False
        print(f"Assume tokenizer {tokenizer.name_or_path} doesn't support system prompt, using system prompt = False")
    
    multi_turn_env = AugmentedQuestionEnv(
        dataset=dataset["train"],
        eval_dataset=dataset["test"],
        support_system_prompt=support_system_prompt,
        system_prompt=training_args.system_prompt,
        task=script_args.task,
        answer_key=script_args.answer_key,
        total_rules=script_args.total_rules,
        max_steps=script_args.max_turns,
        max_tokens=training_args.max_completion_length,    # This is the maximum tokens for each samples used in GRPO
        eot_id=tokenizer.eos_token_id,
        message_end_id=message_end_id,
        message_end_newline_id=message_end_newline_id,
        filter_easy_or_difficult_samples=script_args.filter_easy_or_difficult_samples,
    )

    for split in dataset:
        if "messages" in dataset[split].column_names:
            dataset[split] = dataset[split].remove_columns("messages")

    logger.info("*** Initializing model kwargs ***")
    torch_dtype = (
        model_args.torch_dtype if model_args.torch_dtype in ["auto", None] else getattr(torch, model_args.torch_dtype)
    )
    model_kwargs = dict(
        revision=model_args.model_revision,
        trust_remote_code=model_args.trust_remote_code,
        attn_implementation=model_args.attn_implementation,
        torch_dtype=torch_dtype,
        use_cache=False if training_args.gradient_checkpointing else True,
    )
    training_args.model_init_kwargs = model_kwargs

    #############################
    # Initialize the GRPO trainer
    #############################
    print("SFT Weight: ", training_args.sft_weight)
    trainer = MultiTurnGRPOTrainer(
        model=model_args.model_name_or_path,
        env=multi_turn_env,
        reward_funcs=reward_funcs,
        args=training_args,
        train_dataset=multi_turn_env.get_dataset(),
        eval_dataset=multi_turn_env.get_eval_dataset() if training_args.eval_strategy != "no" else None,
        peft_config=get_peft_config(model_args),
        callbacks=get_callbacks(training_args, model_args),
        processing_class=tokenizer,
    )

    ###############
    # Training loop
    ###############
    logger.info("*** Train ***")
    checkpoint = None
    if training_args.resume_from_checkpoint is not None:
        checkpoint = training_args.resume_from_checkpoint
    elif last_checkpoint is not None:
        checkpoint = last_checkpoint
    train_result = trainer.train(resume_from_checkpoint=checkpoint)
    metrics = train_result.metrics
    metrics["train_samples"] = len(dataset[script_args.dataset_train_split])
    trainer.log_metrics("train", metrics)
    trainer.save_metrics("train", metrics)
    trainer.save_state()

    ##################################
    # Save model and create model card
    ##################################
    logger.info("*** Save model ***")
    trainer.save_model(training_args.output_dir)
    logger.info(f"Model saved to {training_args.output_dir}")

    # Save everything else on main process
    kwargs = {
        "dataset_name": script_args.dataset_name,
        "tags": ["open-r1", "verifiers", "multi-turn", "grpo"],
    }
    if trainer.accelerator.is_main_process:
        trainer.create_model_card(**kwargs)
        # Restore k,v cache for fast inference
        trainer.model.config.use_cache = True
        trainer.model.config.save_pretrained(training_args.output_dir)

    ##########
    # Evaluate
    ##########
    if training_args.do_eval and training_args.eval_strategy != "no":
        logger.info("*** Evaluate ***")
        metrics = trainer.evaluate()
        metrics["eval_samples"] = len(multi_turn_env.get_eval_dataset())
        trainer.log_metrics("eval", metrics)
        trainer.save_metrics("eval", metrics)

    # #############
    # # push to hub
    # #############
    # if training_args.push_to_hub:
    #     logger.info("Pushing to hub...")
    #     trainer.push_to_hub(**kwargs)

    # Clean up distributed training
    if training_args.local_rank != -1:
        import torch.distributed as dist
        dist.destroy_process_group()


if __name__ == "__main__":
    parser = TrlParser((GRPOScriptArguments, GRPOConfig, ModelConfig))
    script_args, training_args, model_args = parser.parse_args_and_config()
    main(script_args, training_args, model_args)
