""" This script loads a base classifier and then runs PREDICT on many examples from a dataset.
"""
import argparse
from datasets import get_dataset, DATASETS, get_num_classes
from torch.utils.data import DataLoader, WeightedRandomSampler
from core import Smooth
from time import time
import torch
from architectures import get_architecture
import datetime
import numpy as np
import torch.backends.cudnn as cudnn
cudnn.benchmark = True
parser = argparse.ArgumentParser(description='Predict on many examples')
parser.add_argument("--dataset", default="AWA",choices=DATASETS, help="which dataset")
parser.add_argument("--base_classifier", type=str, default='main_model_sd025/checkpoint.pth.tar', help="path to saved pytorch model of base classifier")
parser.add_argument("--sigma", type=float,default=0.25, help="noise hyperparameter")
parser.add_argument("--outfile", type=str, default = 'prediction_result/simulation.txt', help="output file")
parser.add_argument("--batch", type=int, default=1000, help="batch size")
parser.add_argument("--skip", type=int, default=1, help="how many examples to skip")
parser.add_argument("--max", type=int, default= -1, help="stop after this many examples")
parser.add_argument("--split", choices=["train", "test"], default="train", help="train or test set")
parser.add_argument("--N", type=int, default=100, help="number of samples to use")
parser.add_argument("--alpha", type=float, default=0.001, help="failure probability")
parser.add_argument('--card', action='store_true', default=False)
parser.add_argument('--model', type=str, default = "reason_model", help='folder to save model and training log)')
args = parser.parse_args()

if __name__ == "__main__":
    # load the base classifier
    if args.card:
        smoothed_classifier = Smooth(None, get_num_classes(args.dataset), args.sigma, args.card, args.model)
    else:
        checkpoint = torch.load(args.base_classifier)
        base_classifier = get_architecture(checkpoint["arch"], args.dataset)
        base_classifier.load_state_dict(checkpoint['state_dict'])

        # create the smooothed classifier g
        smoothed_classifier = Smooth(base_classifier, get_num_classes(args.dataset), args.sigma, args.card)

    # prepare output file
    f = open(args.outfile, 'w')
    print("idx\tlabel\tpredict\tcorrect\ttime", file=f, flush=True)

    # iterate through the dataset
    dataset = get_dataset(args.dataset, args.split)
    for i in range(len(dataset)):

        # only certify every args.skip examples, and stop after args.max examples
        if i % args.skip != 0:
            continue
        if i == args.max:
            break

        (x, label) = dataset[i]
        x = x.cuda()
        before_time = time()

        # make the prediction
        prediction = smoothed_classifier.predict(x, args.N, args.alpha, args.batch)

        after_time = time()
        correct = int(prediction == label)

        time_elapsed = str(datetime.timedelta(seconds=(after_time - before_time)))

        # log the prediction and whether it was correct
        print("{}\t{}\t{}\t{}\t{}".format(i, label, prediction, correct, time_elapsed), file=f, flush=True)

    f.close()