import math
import numpy as np
import tkinter as tk
from PIL import Image, ImageDraw
import os

def calculate_target_cup_water_level(base_cup_width, base_cup_height, base_cup_water_level, target_cup_width, target_cup_height):
    print(f"base_cup_width: {base_cup_width}, base_cup_height: {base_cup_height}, base_cup_water_level: {base_cup_water_level}, target_cup_width: {target_cup_width}, target_cup_height: {target_cup_height}")
    
    # get the angle of tilt for the base cup
    if base_cup_water_level >= base_cup_height / 2:
        # the bottom is fully covered

        # get the average height of the water
        # one side is going to be at the cup height
        # so the other side is going to be at water_level - (cup_height - water_level)
        lower_end_height = 2 * base_cup_water_level - base_cup_height
        air_height = base_cup_height - lower_end_height
        tilt_angle = math.atan(air_height / base_cup_width)
    
    else:
        # the bottom is not fully covered, the entire thing is a triangle
        overall_area = base_cup_width * base_cup_water_level # this is the rectangle area

        # water_width * base_cup_height / 2 = overall area
        water_width = overall_area * 2 / base_cup_height
        tilt_angle = math.atan(base_cup_height / water_width)

    # get the angle of the target cup diagonal
    target_cup_angle = math.atan(target_cup_height / target_cup_width)
    # if the angle is greater than the diagonal, then we need less water than enough to fill half the cup. 
    if tilt_angle >= target_cup_angle:
        # the height is still the height
        # the width is unknown
        # height/width = math.tan(tilt_angle)
        target_water_width_when_tilted = target_cup_height / math.tan(tilt_angle)
        target_water_area = target_cup_height * target_water_width_when_tilted / 2
        target_water_height = target_water_area / target_cup_width
    
    else:
        # the height is still the height, but some of the bottom is completely filled
        # the width of the triangle and the rectangle are known
        # height of triangle/width = math.tan(tilt_angle)
        triangle_height = target_cup_width * math.tan(tilt_angle)
        rectangle_height = target_cup_height - triangle_height
        # height of water is average of rectangle height and cup height
        target_water_height = (rectangle_height + target_cup_height) / 2
    
    return round(target_water_height, 2)


def generate_answer_options(args, correct_target_cup_water_level, target_cup_height):
    # use normal distribution to generate noise, maximum should be the distance to the lip of the cup
    if args.verbose:
        print(f"generating answer options for correct_target_cup_water_level: {correct_target_cup_water_level}, target_cup_height: {target_cup_height}")
    distance_to_lip = target_cup_height - correct_target_cup_water_level
    answers = [correct_target_cup_water_level]

    while len(answers) < 4:
        new_noise = np.random.normal(0, distance_to_lip/2)
        potential_answer = correct_target_cup_water_level - new_noise
        potential_answer = round(potential_answer, 2)
        valid = True

        if potential_answer > target_cup_height or potential_answer < 0:
            valid = False
        
        for answer in answers:
            if abs(potential_answer - answer) < args.difficulty_margin:
                valid = False
        
        if valid:
            answers.append(potential_answer)
    
    return answers


def draw_cups(args, dataset):
    # Initialize the Tkinter root window
    root = tk.Tk()
    root.withdraw()
    # Create a canvas to draw on
    canvas_width = 500
    canvas_height = 500
    canvas = tk.Canvas(root, width=canvas_width, height=canvas_height)
    canvas.pack()

    for i in range(len(dataset)):
        # Clear the canvas
        canvas.delete("all")

        problem = dataset[i]
        problem_image_path, correct_label = draw_cup(
            args, i, canvas, canvas_width, canvas_height, \
            problem["base_cup_width"], problem["base_cup_height"], problem["base_cup_water_level"], \
            problem["target_cup_width"], problem["target_cup_height"], \
            problem["answer_options"], problem["correct_target_cup_water_level"]
        )
        problem["problem_image_path"] = problem_image_path
        problem["correct_label"] = correct_label

        dataset[i] = problem
    
    root.destroy()
    return dataset


def draw_cup(args, i, canvas, canvas_width, canvas_height, base_cup_width, base_cup_height, base_cup_water_level, target_cup_width, target_cup_height, answers, correct_target_cup_water_level):

    if args.verbose:
        print(f"drawing base_cup_width: {base_cup_width}, base_cup_height: {base_cup_height}, base_cup_water_level: {base_cup_water_level}, target_cup_width: {target_cup_width}, target_cup_height: {target_cup_height}, answers: {answers}, correct_target_cup_water_level: {correct_target_cup_water_level}")

    draw_base_cup_width, draw_base_cup_height, draw_base_cup_water_level = base_cup_width * args.drawing_scale, base_cup_height * args.drawing_scale, base_cup_water_level * args.drawing_scale
    draw_target_cup_width, draw_target_cup_height = target_cup_width * args.drawing_scale, target_cup_height * args.drawing_scale
    answers = [answer * args.drawing_scale for answer in answers]

    left_center_point, left_center_height = canvas_width/4, canvas_height/2
    right_center_point, right_center_height = canvas_width/4*3, canvas_height/2

    # identify the table y
    table_y = max(left_center_height + draw_base_cup_height, right_center_height + draw_target_cup_height)

    # draw the base cup
    base_cup_left_top_x = left_center_point - draw_base_cup_width/2
    base_cup_left_top_y = table_y - draw_base_cup_height
    base_cup_right_bottom_x = base_cup_left_top_x + draw_base_cup_width
    base_cup_right_bottom_y = table_y

    # draw the water level on the base cup as a line
    water_level_height = base_cup_right_bottom_y - draw_base_cup_water_level
    water_level_left = base_cup_left_top_x
    water_level_right = base_cup_right_bottom_x

    # draw the target cup
    target_cup_left_top_x = right_center_point - draw_target_cup_width/2
    target_cup_left_top_y = table_y - draw_target_cup_height
    target_cup_right_bottom_x = target_cup_left_top_x + draw_target_cup_width
    target_cup_right_bottom_y = table_y

    # draw the cups
    canvas.create_rectangle(base_cup_left_top_x, base_cup_left_top_y, base_cup_right_bottom_x, base_cup_right_bottom_y, outline="black", fill="white")
    canvas.create_rectangle(target_cup_left_top_x, target_cup_left_top_y, target_cup_right_bottom_x, target_cup_right_bottom_y, outline="black", fill="white")
    canvas.create_line(water_level_left, water_level_height, water_level_right, water_level_height, fill="blue")

    # draw the answers
    answer_ticks_left_x = target_cup_right_bottom_x
    answer_ticks_right_x = target_cup_right_bottom_x + 10
    for answer in answers:
        answer_tick_y = table_y - answer
        canvas.create_line(answer_ticks_left_x, answer_tick_y, answer_ticks_right_x, answer_tick_y, fill="black")

        index = 3 - sorted(answers).index(answer)
        # map to A B C D
        answer_label = chr(65 + index)

        if answer == correct_target_cup_water_level * args.drawing_scale:
            correct_label = answer_label
        
        canvas.create_text(answer_ticks_left_x + 20, answer_tick_y, text=answer_label, fill="black", font=("Arial", 13))

    canvas.update()

    # I want to save the canvas to a png file
    canvas.postscript(file="canvas_output.eps")
    with Image.open("canvas_output.eps") as image:
        path = f"{args.dataset_dir}/{i}.png"
        image.save(path, "png")

    # Remove the temporary EPS file
    os.remove("canvas_output.eps")

    return path, correct_label