import math
import numpy as np
import torch
import torchvision.transforms as T
# from decord import VideoReader, cpu
from PIL import Image
from torchvision.transforms.functional import InterpolationMode
from transformers import AutoModel, AutoTokenizer
from pynvml import (
    nvmlInit,
    nvmlDeviceGetHandleByIndex,
    nvmlDeviceGetMemoryInfo,
)
from tqdm import tqdm
from rich import print
from pathlib import Path
import json


IMAGENET_MEAN = (0.485, 0.456, 0.406)
IMAGENET_STD = (0.229, 0.224, 0.225)


def build_transform(input_size):
    MEAN, STD = IMAGENET_MEAN, IMAGENET_STD
    transform = T.Compose([
        T.Lambda(lambda img: img.convert('RGB') if img.mode != 'RGB' else img),
        T.Resize((input_size, input_size), interpolation=InterpolationMode.BICUBIC),
        T.ToTensor(),
        T.Normalize(mean=MEAN, std=STD)
    ])
    return transform


def find_closest_aspect_ratio(aspect_ratio, target_ratios, width, height, image_size):
    best_ratio_diff = float('inf')
    best_ratio = (1, 1)
    area = width * height
    for ratio in target_ratios:
        target_aspect_ratio = ratio[0] / ratio[1]
        ratio_diff = abs(aspect_ratio - target_aspect_ratio)
        if ratio_diff < best_ratio_diff:
            best_ratio_diff = ratio_diff
            best_ratio = ratio
        elif ratio_diff == best_ratio_diff:
            if area > 0.5 * image_size * image_size * ratio[0] * ratio[1]:
                best_ratio = ratio
    return best_ratio


def dynamic_preprocess(image, min_num=1, max_num=12, image_size=448, use_thumbnail=False):
    orig_width, orig_height = image.size
    aspect_ratio = orig_width / orig_height

    # calculate the existing image aspect ratio
    target_ratios = set(
        (i, j) for n in range(min_num, max_num + 1) for i in range(1, n + 1) for j in range(1, n + 1) if
        i * j <= max_num and i * j >= min_num)
    target_ratios = sorted(target_ratios, key=lambda x: x[0] * x[1])

    # find the closest aspect ratio to the target
    target_aspect_ratio = find_closest_aspect_ratio(
        aspect_ratio, target_ratios, orig_width, orig_height, image_size)

    # calculate the target width and height
    target_width = image_size * target_aspect_ratio[0]
    target_height = image_size * target_aspect_ratio[1]
    blocks = target_aspect_ratio[0] * target_aspect_ratio[1]

    # resize the image
    resized_img = image.resize((target_width, target_height))
    processed_images = []
    for i in range(blocks):
        box = (
            (i % (target_width // image_size)) * image_size,
            (i // (target_width // image_size)) * image_size,
            ((i % (target_width // image_size)) + 1) * image_size,
            ((i // (target_width // image_size)) + 1) * image_size
        )
        # split the image
        split_img = resized_img.crop(box)
        processed_images.append(split_img)
    assert len(processed_images) == blocks
    if use_thumbnail and len(processed_images) != 1:
        thumbnail_img = image.resize((image_size, image_size))
        processed_images.append(thumbnail_img)
    return processed_images


def load_image(image_file, input_size=448, max_num=12):
    image = Image.open(image_file).convert('RGB')
    transform = build_transform(input_size=input_size)
    images = dynamic_preprocess(image, image_size=input_size, use_thumbnail=True, max_num=max_num)
    pixel_values = [transform(image) for image in images]
    pixel_values = torch.stack(pixel_values)
    return pixel_values


def split_model(model_name):
    device_map = {}
    world_size = torch.cuda.device_count()
    num_layers = {
        'InternVL2-1B': 24, 'InternVL2-2B': 24, 'InternVL2-4B': 32, 'InternVL2-8B': 32,
        'InternVL2-26B': 48, 'InternVL2-40B': 60, 'InternVL2-Llama3-76B': 80}[model_name]
    # Since the first GPU will be used for ViT, treat it as half a GPU.
    num_layers_per_gpu = math.ceil(num_layers / (world_size - 0.5))
    num_layers_per_gpu = [num_layers_per_gpu] * world_size
    num_layers_per_gpu[0] = math.ceil(num_layers_per_gpu[0] * 0.5)
    layer_cnt = 0
    for i, num_layer in enumerate(num_layers_per_gpu):
        for j in range(num_layer):
            device_map[f'language_model.model.layers.{layer_cnt}'] = i
            layer_cnt += 1
    device_map['vision_model'] = 0
    device_map['mlp1'] = 0
    device_map['language_model.model.rotary_emb']= 0
    device_map['language_model.model.tok_embeddings'] = 0
    device_map['language_model.model.embed_tokens'] = 0
    device_map['language_model.output'] = 0
    device_map['language_model.model.norm'] = 0
    device_map['language_model.lm_head'] = 0
    device_map[f'language_model.model.layers.{num_layers - 1}'] = 0
    return device_map


def print_gpu_usage():
    nvmlInit()
    n_gpus = torch.cuda.device_count()

    print('========== GPU Utilization ==========')
    for gpu_id in range(n_gpus):
        h = nvmlDeviceGetHandleByIndex(gpu_id)
        info = nvmlDeviceGetMemoryInfo(h)
        print(f'GPU {gpu_id}')
        print(f'- Used:       {info.used / 1024 ** 3:>8.2f} B ({info.used / info.total * 100:.1f}%)')
        print(f'- Available:  {info.free / 1024 ** 3:>8.2f} B ({info.free / info.total * 100:.1f}%)')
        print(f'- Total:      {info.total / 1024 ** 3:>8.2f} B')
    print('=====================================')
    

def infer(pixel_values, question):
    path = 'redacted'
    device_map = split_model('InternVL2-40B')
    model = AutoModel.from_pretrained(
        path,
        torch_dtype=torch.bfloat16,
        # load_in_8bit=True,
        low_cpu_mem_usage=True,
        use_flash_attn=True,
        trust_remote_code=True,
        device_map=device_map,
        ).eval()
    tokenizer = AutoTokenizer.from_pretrained(path, trust_remote_code=True, use_fast=False)
    generation_config = dict(max_new_tokens=500, do_sample=True)

    response = model.chat(tokenizer, pixel_values, question, generation_config)
    print(f"response: {response}")
    print_gpu_usage()
    
    # del model
    torch.cuda.empty_cache()
    del model
    return response
    
    
if __name__ == "__main__":
    data_path = f"redacted/stable-image-ultra"
    
    for idx in tqdm(range(100), desc='Inferencing'):
        save_path = f'redacted/results/InternalVL2-40B/{idx}'
        Path(save_path).mkdir(parents=True, exist_ok=True)
        save_file = f'{save_path}/output.json'
        
        pixel_lists = []
        pixel_values1 = load_image(f'{data_path}/{str(idx)}/double_left.png', max_num=12).to(torch.bfloat16).cuda()
        
        correct_image_file = f'{data_path}/{str(idx)}/double_right.png'
        inserting_index = np.random.randint(1, 6)
        indiecs = [0, 1, 2, 3]
        for i in range(6):
            if i == 0:
                pixel_lists.append(pixel_values1)
            else:
                if i == inserting_index:
                    pixel_values = load_image(correct_image_file, max_num=12).to(torch.bfloat16).cuda()
                else:
                    index = indiecs.pop(0)
                    pixel_values = load_image(f'{data_path}/{str(idx)}/single_{index}.png', max_num=12).to(torch.bfloat16).cuda()
                pixel_lists.append(pixel_values)
        
        pixel_values = torch.cat(pixel_lists, dim=0)
        
        zero_shot_question = "Here is an image of a person.\nImage-0: <image>\nFrom the images below, which image contains the same person as the person in Image-0.\n"
        for idx_img in range(5):
            zero_shot_question += f"Image-{idx_img+1}: <image>\n"
        zero_shot_question += "Please answer with the image number.\n"
        
        zero_shot_response = infer(pixel_values, zero_shot_question)

        cot_question = "Here is an image of a person.\nImage-0: <image>\nFrom the images below, which image contains the same person as the person in Image-0.\n"
        for idx_img in range(5):
            cot_question += f"Image-{idx_img+1}: <image>\n"
        cot_question += "Please think step-by-step before answering the image number.\n"
        cot_response = infer(pixel_values, cot_question)
        
        output = {
            'zero_shot_question': zero_shot_question,
            'cot_question': cot_question,
            'zero_shot_response': zero_shot_response,
            'cot_response': cot_response,
            'correct_image': inserting_index
        }
        with open(save_file, 'w') as f:
            json.dump(output, f, indent=4)
            
        
