

from pathlib import Path
import argparse, json, numpy as np, pandas as pd, torch
from datasets import Dataset
from transformers import (
    BigBirdTokenizerFast, BigBirdModel,
    TrainingArguments, Trainer, DefaultDataCollator
)
from torch import nn
from transformers import EarlyStoppingCallback
import evaluate

LABEL2ID = {"NotExtrapolated": 0, "Partial": 1, "Full": 2, "Unlabeled": 3}
ID2LABEL = {v: k for k, v in LABEL2ID.items()}

class BigBirdLinearProbe(nn.Module):
    def __init__(self, model_name="google/bigbird-roberta-base", num_labels=4):
        super().__init__()
        self.encoder = BigBirdModel.from_pretrained(model_name)
        for param in self.encoder.parameters():
            param.requires_grad = False  # Freeze encoder
        hidden_size = self.encoder.config.hidden_size
        self.classifier = nn.Linear(hidden_size, num_labels)

    def forward(self, input_ids, attention_mask, labels=None):
        outputs = self.encoder(input_ids=input_ids, attention_mask=attention_mask)
        cls_rep = outputs.last_hidden_state[:, 0, :]  # CLS token
        logits = self.classifier(cls_rep)

        if labels is not None:
            loss_fn = nn.CrossEntropyLoss()
            loss = loss_fn(logits, labels)
            return {"loss": loss, "logits": logits}
        return {"logits": logits}

def _canon_txt(txt_dir: Path, cid: str) -> Path:
    return txt_dir / f"{cid}.txt"

def _filter_ok(df: pd.DataFrame, txt_dir: Path) -> pd.DataFrame:
    df = df.dropna(subset=["canon_id"]).copy()
    df["txt_file"] = df["canon_id"].apply(lambda c: _canon_txt(txt_dir, c))
    df = df[df["txt_file"].apply(Path.exists)]
    df["label"] = df["label"].replace("", "NotExtrapolated").map(LABEL2ID).astype(int)
    df["txt_file"] = df["txt_file"].astype(str)
    assert len(df), "No rows left after matching txt files – check paths!"
    return df[["txt_file", "label"]]

def load_splits(split_dir: Path, txt_dir: Path):
    dfs = [pd.read_csv(split_dir / f"{s}.csv", dtype=str) for s in ["train","dev","test"]]
    return [Dataset.from_pandas(_filter_ok(df, txt_dir)) for df in dfs]

def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--split_dir", required=True, type=Path)
    ap.add_argument("--txt_dir", required=True, type=Path)
    ap.add_argument("--out_dir", default=Path("checkpoints/bigbird_linear_probe"), type=Path)
    ap.add_argument("--epochs", default=4, type=int)
    args = ap.parse_args()

    ds_train, ds_dev, ds_test = load_splits(args.split_dir, args.txt_dir)

    tokenizer = BigBirdTokenizerFast.from_pretrained("google/bigbird-roberta-base")
    def tok_fn(batch):
        texts = [Path(p).read_text(errors="ignore") for p in batch["txt_file"]]
        enc = tokenizer(texts, max_length=4096, truncation=True, padding="max_length")
        enc["labels"] = batch["label"]
        return enc

    ds_train = ds_train.map(tok_fn, batched=True, remove_columns=["txt_file", "label"]).with_format("torch")
    ds_dev   = ds_dev.map(tok_fn, batched=True, remove_columns=["txt_file", "label"]).with_format("torch")
    ds_test  = ds_test.map(tok_fn, batched=True, remove_columns=["txt_file", "label"]).with_format("torch")

    model = BigBirdLinearProbe()

    metrics = {
        "accuracy": evaluate.load("accuracy"),
        "f1": evaluate.load("f1"),
    }

    def compute_metrics(eval_pred):
        logits, labels = eval_pred
        preds = np.argmax(logits, axis=1)
        return {
            "accuracy": metrics["accuracy"].compute(predictions=preds, references=labels),
            "macro_f1": metrics["f1"].compute(predictions=preds, references=labels, average="macro"),
            "weighted_f1": metrics["f1"].compute(predictions=preds, references=labels, average="weighted"),
        }

    training_args = TrainingArguments(
        output_dir=args.out_dir,
        per_device_train_batch_size=1,
        gradient_accumulation_steps=8,
        fp16=True,
        learning_rate=2e-5,
        seed=42,
        eval_strategy="steps",
        eval_steps=100,
        save_strategy="steps",
        save_steps=1000,
        load_best_model_at_end=True,
        metric_for_best_model="eval_loss",
        greater_is_better=False,
        logging_steps=50,
        num_train_epochs=args.epochs,
    )

    trainer = Trainer(
        model=model,
        args=training_args,
        train_dataset=ds_train,
        eval_dataset=ds_dev,
        data_collator=DefaultDataCollator(return_tensors="pt"),
        compute_metrics=compute_metrics,
        callbacks=[EarlyStoppingCallback(early_stopping_patience=2)]
    )

    trainer.train()
    metrics_test = trainer.evaluate(ds_test, metric_key_prefix="test")
    print(json.dumps(metrics_test, indent=2))

    args.out_dir.mkdir(parents=True, exist_ok=True)
    torch.save(model.state_dict(), args.out_dir / "linear_probe_model.pt")
    tokenizer.save_pretrained(args.out_dir)
    (args.out_dir / "test_metrics.json").write_text(json.dumps(metrics_test, indent=2))

if __name__ == "__main__":
    main()

