#!/usr/bin/env python
"""
Fine-tune BioClinicalBERT (emilyalsentzer/Bio_ClinicalBERT) on PED-X-Bench
with **4 classes** (including “Unlabeled”).

Metrics reported: accuracy, macro-F1, weighted-F1, macro AUC.
"""

from pathlib import Path
import argparse, json, numpy as np, pandas as pd, torch
from datasets import Dataset
from transformers import (
    AutoTokenizer,
    AutoModelForSequenceClassification,
    TrainingArguments,
    Trainer,
    DefaultDataCollator,
    EarlyStoppingCallback,
)
import evaluate
from sklearn.metrics import roc_auc_score

# ---------------------------------------------------------------------
LABEL2ID = {
    "NotExtrapolated": 0,
    "Partial"        : 1,
    "Full"           : 2,
    "Unlabeled"      : 3,
}
ID2LABEL = {v: k for k, v in LABEL2ID.items()}

# ---------------------------------------------------------------------
def _canon_txt(txt_dir: Path, cid: str) -> Path:
    return txt_dir / f"{cid}.txt"

def _filter_ok(df: pd.DataFrame, txt_dir: Path) -> pd.DataFrame:
    """Keep rows whose txt file exists; keep all 4 labels."""
    df = df.dropna(subset=["canon_id"]).copy()
    df["txt_file"] = df["canon_id"].apply(lambda c: _canon_txt(txt_dir, c))
    df = df[df["txt_file"].apply(Path.exists)]

    # fill empty label → majority class
    df["label"] = df["label"].replace("", "NotExtrapolated").map(LABEL2ID)
    df = df.astype({"label": "int32"})
    df["txt_file"] = df["txt_file"].astype(str)
    assert len(df), "No rows left after filtering!"
    return df[["txt_file", "label"]]

def load_splits(split_dir: Path, txt_dir: Path):
    dfs = [pd.read_csv(split_dir / f"{s}.csv", dtype=str) for s in ["train", "dev", "test"]]
    return [Dataset.from_pandas(_filter_ok(df, txt_dir)) for df in dfs]

# ---------------------------------------------------------------------
def main():
    ap = argparse.ArgumentParser()
    ap.add_argument("--split_dir", required=True, type=Path)
    ap.add_argument("--txt_dir",   required=True, type=Path)
    ap.add_argument("--out_dir",   default=Path("checkpoints/bioclinicalbert_4cls"), type=Path)
    ap.add_argument("--epochs",    default=10, type=int)
    args = ap.parse_args()

    ds_train, ds_dev, ds_test = load_splits(args.split_dir, args.txt_dir)

    tok = AutoTokenizer.from_pretrained("emilyalsentzer/Bio_ClinicalBERT")

    def tok_fn(batch):
        texts = [Path(p).read_text(errors="ignore") for p in batch["txt_file"]]
        enc = tok(texts, max_length=512, truncation=True, padding="max_length")
        enc["labels"] = batch["label"]
        return enc

    ds_train = ds_train.map(tok_fn, batched=True, remove_columns=["txt_file","label"]).with_format("torch")
    ds_dev   = ds_dev  .map(tok_fn, batched=True, remove_columns=["txt_file","label"]).with_format("torch")
    ds_test  = ds_test .map(tok_fn, batched=True, remove_columns=["txt_file","label"]).with_format("torch")

    model = AutoModelForSequenceClassification.from_pretrained(
        "emilyalsentzer/Bio_ClinicalBERT",
        num_labels=4,
        label2id=LABEL2ID,
        id2label=ID2LABEL,
    )

    # ---------------- metrics ----------------
    hf_acc = evaluate.load("accuracy")
    hf_f1  = evaluate.load("f1")

    def compute(eval_pred):
        logits, labels = eval_pred
        preds = np.argmax(logits, axis=1)
        probs = torch.softmax(torch.tensor(logits), dim=1).cpu().numpy()   # (N,4)

        labels_np = labels.astype(np.int32)
        one_hot   = np.eye(4, dtype=np.int32)[labels_np]

        return {
            "accuracy"   : hf_acc.compute(predictions=preds, references=labels_np)["accuracy"],
            "macro_f1"   : hf_f1.compute(predictions=preds, references=labels_np, average="macro")["f1"],
            "weighted_f1": hf_f1.compute(predictions=preds, references=labels_np, average="weighted")["f1"],
            "auc"        : roc_auc_score(one_hot, probs, average="macro", multi_class="ovr"),
        }

    # ---------------- training args ----------------
    training_args = TrainingArguments(
        output_dir=args.out_dir,
        per_device_train_batch_size=4,
        gradient_accumulation_steps=2,
        learning_rate=2e-5,
        num_train_epochs=args.epochs,
        eval_strategy="steps",
        eval_steps=100,
        save_strategy="steps",
        save_steps=1000,
        load_best_model_at_end=True,
        metric_for_best_model="eval_loss",
        greater_is_better=False,
        fp16=True,      # disable if AMP issues
        seed=42,
        logging_steps=50,
    )

    trainer = Trainer(
        model=model,
        args=training_args,
        train_dataset=ds_train,
        eval_dataset=ds_dev,
        data_collator=DefaultDataCollator(return_tensors="pt"),
        compute_metrics=compute,
        callbacks=[EarlyStoppingCallback(early_stopping_patience=2)],
    )

    trainer.train()
    metrics_test = trainer.evaluate(ds_test, metric_key_prefix="test")
    print(json.dumps(metrics_test, indent=2))

    # save artefacts
    args.out_dir.mkdir(parents=True, exist_ok=True)
    trainer.save_model(args.out_dir)
    tok.save_pretrained(args.out_dir)
    (args.out_dir / "test_metrics.json").write_text(json.dumps(metrics_test, indent=2))

if __name__ == "__main__":
    main()
