from .utils import get_scale, M_Q, flatten_layer, transpose_depthwise
import numpy as np
from typing import Dict, Any, Tuple


def MQ_dense(W: np.ndarray, bits: int) -> Tuple[np.ndarray, np.ndarray, bool]:
    """
    This function works over dense/fully-connected layers
    """
    scale, quantization_range, output_scale = get_scale(W=W, bits=bits)
    if quantization_range < 0:
        return None, None, False
    new_W = M_Q(W=W, s=scale)
    return new_W, output_scale, True


def MQ_conv(W: np.ndarray, bits: int) -> Tuple[np.ndarray, np.ndarray, bool]:
    """
    This function works over convolutional layers that are not depthwise layers.
    This function leverages the fully-connected primitive.
    """
    original_shape = W.shape
    new_W, scale, quantized = MQ_dense(W=flatten_layer(W), bits=bits)
    if quantized:
        new_W = np.reshape(new_W, newshape=(original_shape))
    return new_W, scale, quantized


def MQ_depthwise(W: np.ndarray, bits: int) -> Tuple[np.ndarray, np.ndarray, bool]:
    """
    This function works over convolutional layers that are depthwise layers.
    This function leverages the convolution primitive.
    """
    new_W, scale, quantized = MQ_conv(W=transpose_depthwise(W=W), bits=bits)
    if quantized:
        new_W = transpose_depthwise(W=new_W)
    return new_W, scale, quantized


def M_Quant(W: np.ndarray, layer_type: str, bits: int) -> Dict[str, Any]:
    """
    This function fetches the appropriate function based on the layer_type argument
    It returns the processed weights, scale and confirms the quantization
    """
    if layer_type == "Conv2D":
        new_W, scale, quantized = MQ_conv(W=W, bits=bits)
    elif layer_type == "Dense":
        new_W, scale, quantized = MQ_dense(W=W, bits=bits)
    elif layer_type == "DepthwiseConv2D":
        new_W, scale, quantized = MQ_depthwise(W=W, bits=bits)
    else:
        print(f"[WARNING] in M Quant : unsupported layer type : {layer_type}")
        new_W = None
        scale = None
        quantized = False
    return {
        "original weights": W,
        "quantized weights": new_W,
        "scale": scale,
        "quantized": quantized,
    }
