import torch
import torch.nn as nn
from torch.nn import functional as F


class AE(nn.Module) : 
    def __init__(self, n_dim=1, m_dim=1, nu=3, recon_sigma=1, reg_weight=1, num_hidden=64, device='cpu'):
        super(AE, self).__init__()
        self.model_name = "AE"

        self.n_dim = n_dim
        self.m_dim = m_dim
        self.recon_sigma = recon_sigma
        self.reg_weight = reg_weight
        self.num_hidden = num_hidden
        self.device = device

        # define encoder
        self.encoder = nn.Sequential(
            nn.Linear(n_dim, num_hidden),
            nn.LeakyReLU(),  
            nn.BatchNorm1d(num_hidden),
            # nn.Linear(num_hidden, num_hidden),
            # nn.LeakyReLU(),  
            # nn.BatchNorm1d(num_hidden),
        )
        self.latent_z = nn.Linear(num_hidden, m_dim)

        # define decoder

        self.decoder = nn.Sequential(
            nn.Linear(m_dim, num_hidden), 
            nn.LeakyReLU(), 
            nn.BatchNorm1d(num_hidden),
            nn.Linear(num_hidden, n_dim),
        )


    def encoder_reparameterize(self, mu, logvar) : 
        std = torch.exp(0.5 * logvar)
        eps = torch.randn_like(std)
        return mu + std * eps
    
    def encode(self, x) : 
        x = self.encoder(x)
        z = self.latent_z(x)

        return z
    
    def decode(self, z) : 
        return self.decoder(z)
    
    def recon_loss(self, x, recon_x) : 
        return F.mse_loss(recon_x, x, reduction = 'none').sum(dim = 1).mean(dim = 0) / self.recon_sigma**2

    
    def total_loss(self, x, recon_x) : 
        recon = self.recon_loss(x, recon_x)
        return recon, 0, recon
    

    def generate(self, N = 1000) : 
        prior = torch.randn(N, self.m_dim).to(self.device)
        return self.decode(prior)
    
    def recon_data(self, x) : 
        enc_z = self.encode(x)
        recon_x = self.decode(enc_z)
        return recon_x


    def forward(self, x, verbose = False) : 
        enc_z = self.encode(x)
        recon_x = self.decode(enc_z)
        total_loss = self.total_loss(x, recon_x)
        if verbose is False : 
            return self.total_loss(x, recon_x)
        else : 
            return [
                total_loss, 
                0,
                0,
                0
            ]

        


