import numpy as np
import torch
import torch.nn as nn
from torch.nn import functional as F
from utils.functions import log_t_normalizing_const, gamma_regularizer

class t3VAE(nn.Module) : 
    
    def __init__(self, n_dim=1, m_dim=1, nu=3, recon_sigma=1, reg_weight=1, num_hidden=64, device='cpu'):
        super(t3VAE, self).__init__()
        # self.model_name = f"t3VAE_fixed_nu_{nu}"
        self.model_name = f"t3VAE_nu_{nu}"

        self.n_dim = n_dim
        self.m_dim = m_dim
        self.nu = nu
        self.recon_sigma = recon_sigma
        self.reg_weight = reg_weight
        self.num_hidden = num_hidden
        self.device = device

        self.gamma = -2 / (self.nu + self.n_dim + self.m_dim)
        log_tau_base = -n_dim * np.log(recon_sigma) + log_t_normalizing_const(nu, n_dim) - np.log(nu + n_dim - 2) + np.log(nu-2)
        
        const_2bar1_term_1 = (1 + m_dim / (nu + n_dim -2))
        const_2bar1_term_2_log = -self.gamma / (1+self.gamma) * log_tau_base
        self.const_2bar1 = const_2bar1_term_1 * const_2bar1_term_2_log.exp()
        
        self.log_tau = 2 / (nu + n_dim - 2 ) * log_tau_base
        self.tau_sq = (self.nu / (nu+self.n_dim)) * self.log_tau.exp()

        log_c2 = (np.log(1 + (n_dim+m_dim)/(nu-2)) + n_dim * np.log(recon_sigma) -
        log_t_normalizing_const(nu, n_dim+m_dim)) * (-self.gamma / (1+self.gamma))
        alpha = -self.gamma * nu / (2 * torch.exp(log_c2))
        print(f"alpha : {alpha:.3f}")
        print(f"tau : {self.tau_sq.sqrt():.3f}")

        # define encoder
        self.encoder = nn.Sequential(
            nn.Linear(n_dim, num_hidden),
            nn.LeakyReLU(),  
            nn.BatchNorm1d(num_hidden),
            # nn.Linear(num_hidden, num_hidden),
            # nn.LeakyReLU(),  
            # nn.BatchNorm1d(num_hidden),
        )
        self.latent_mu = nn.Linear(num_hidden, m_dim)
        self.latent_logvar = nn.Linear(num_hidden, m_dim)

        # define decoder

        self.decoder = nn.Sequential(
            nn.Linear(m_dim, num_hidden), 
            nn.LeakyReLU(), 
            nn.BatchNorm1d(num_hidden),
            nn.Linear(num_hidden, n_dim),
        )


    def encoder_reparameterize(self, mu, logvar) : 
        N_sample = mu.shape[0]
        nu_n = self.nu + self.n_dim

        std = np.sqrt(self.nu / nu_n) * torch.exp(0.5 * logvar)
        eps = torch.randn_like(std)

        chi_dist = torch.distributions.chi2.Chi2(torch.tensor([nu_n]))
        v = chi_dist.sample(torch.tensor([N_sample])).to(self.device)

        return mu + std * eps * torch.sqrt(nu_n / v)
    
    def encode(self, x) : 
        x = self.encoder(x)
        mu = self.latent_mu(x)
        logvar = self.latent_logvar(x)
        z = self.encoder_reparameterize(mu, logvar)

        return z, mu, logvar
    
    def decode(self, z) : 
        return self.decoder(z)
    
    def recon_loss(self, x, recon_x) : 
        return F.mse_loss(recon_x, x, reduction = 'none').sum(dim = 1).mean(dim = 0) / self.recon_sigma**2

    def reg_loss(self, mu, logvar) : 
        # return gamma regularizer including constant term
        return gamma_regularizer(mu, logvar, self.n_dim, self.const_2bar1, self.gamma, self.tau_sq, self.nu)
    
    def total_loss(self, x, recon_x, mu, logvar) : 
        recon = self.recon_loss(recon_x, x)
        reg = self.reg_loss(mu, logvar)

        return recon, reg, recon + self.reg_weight * reg


    def generate(self, N = 1000):
        log_tau = self.log_tau.expand(N, self.m_dim).to(self.device)
        mu = torch.zeros(N, self.m_dim).to(self.device).to(self.device)
        eps = self.encoder_reparameterize(mu, log_tau)
        return self.decode(eps)
    
    def recon_data(self, x) : 
        enc_z, _, _ = self.encode(x)
        recon_x = self.decode(enc_z)
        return recon_x

    def forward(self, x, verbose = False) : 
        enc_z, mu, logvar = self.encode(x)
        recon_x = self.decode(enc_z)
        total_loss = self.total_loss(x, recon_x, mu, logvar)
        if verbose is False : 
            return total_loss
        else : 
            return [
                total_loss, 
                mu.detach().flatten().cpu().numpy(), 
                logvar.detach().flatten().exp().cpu().numpy(), 
                -self.ELBO(x, enc_z, mu, logvar)
            ]
        

        
        

