import re

def answer_to_latex(answer):
    """
    Convert a math answer to LaTeX format, returning False if not possible.
    
    Generated by Claude, doesn't work great.
    """
    
    # If input is not a string, return False
    if not isinstance(answer, str):
        return False
    
    # Strip whitespace and convert to lowercase for processing
    answer = answer.strip().lower()
    
    # Check for long strings without mathematical operators
    if len(answer.replace(' ', '')) > 15 and not any(c in answer for c in '+-*/^√()=<>≤≥'):
        return False
    
    # List of common words that indicate non-mathematical answers
    non_math_words = {
        'look', 'for', 'not', 'applicable', 'undefined', 'grade', 'video', 'games',
        'chocolate', 'oranges', 'constant', 'charlie', 'more', 'complementary',
        'supplementary', 'acute', 'obtuse', 'reflex', 'straight', 'irregular',
        'concave', 'regular', 'identify', 'shortest', 'tallest', 'bar', 'product',
        'with', 'the', 'angle', 'value', 'true', 'false', 'no', 'yes', 'graph',
        'region', 'above', 'below', 'line', 'width', 'length'
    }
    
    # Check for non-mathematical words, but exclude 'and' and 'or' as they might be logical operators
    words = answer.replace('-', ' ').split()
    if any(word in non_math_words for word in words):
        return False
    
    # Enhanced geometric shape detection
    shape_keywords = {
        'triangle', 'square', 'rectangle', 'pentagon', 'hexagon', 'heptagon',
        'octagon', 'nonagon', 'decagon', 'polygon', 'quadrilateral',
        'parallelogram', 'rhombus', 'trapezoid', 'kite'
    }
    
    # Remove spaces and check for combined words containing shape names
    spaceless_answer = answer.replace(' ', '')
    if any(shape in spaceless_answer for shape in shape_keywords):
        return False
    
    # Check original answer with spaces for multi-word shapes
    if any(shape in answer for shape in shape_keywords):
        return False
    
    # Remove units and extra text
    units = r'\s*(cm|m|ft|in|km|meters|degrees|radians|hour|bills|marbles|nickels|dimes|cups|boys|girls|furniture|pencils|erasers|lions|tigers|\$|°|miles per hour).*$'
    answer = re.sub(units, '', answer, flags=re.IGNORECASE)
    
    # Convert logical operators to LaTeX
    def convert_logical_operators(expr):
        parts = expr.split()
        for i, part in enumerate(parts):
            if part == 'and':
                parts[i] = r'\land'
            elif part == 'or':
                parts[i] = r'\lor'
        return ' '.join(parts)
    
    # Handle inequalities with multiple conditions and logical operators
    if any(op in answer for op in [' and ', ' or ']) and any(x in answer for x in ['<', '>', '≤', '≥']):
        # Split on 'and' and 'or', preserving the operator
        parts = []
        current = ''
        for word in answer.split():
            if word in ['and', 'or']:
                if current:
                    parts.append(current.strip())
                parts.append(word)
                current = ''
            else:
                current += ' ' + word if current else word
        if current:
            parts.append(current.strip())
        
        # Process each part
        processed_parts = []
        for part in parts:
            part = part.strip()
            if part in ['and', 'or']:
                processed_parts.append(r'\land' if part == 'and' else r'\lor')
            else:
                part = part.replace('≤', r'\leq').replace('≥', r'\geq')
                processed_parts.append(part)
        
        return ' '.join(processed_parts)
    
    # Handle simple inequalities
    if '<' in answer or '>' in answer or '≤' in answer or '≥' in answer:
        answer = answer.replace('≤', r'\leq').replace('≥', r'\geq')
        return convert_logical_operators(answer)
    
    # Handle solutions with multiple variables
    if answer.startswith('x=') and 'y=' in answer:
        parts = answer.split(',')
        if len(parts) == 2:
            x_val = parts[0].replace('x=', '').strip()
            y_val = parts[1].replace('y=', '').strip()
            return f"(x={x_val}, y={y_val})"
    
    # Handle fractions
    if '/' in answer:
        parts = answer.split('/', 1)
        if len(parts) == 2:
            num, denom = parts
            if '√' in num:
                if num.startswith('√'):
                    return f"\\frac{{\\sqrt{{{num[1:]}}}}}{{\\{denom}}}"
                return f"\\frac{{{num}}}{{{denom}}}"
            if all(re.match(r'^-?\d+$', n.strip()) for n in [num, denom]):
                return f"\\frac{{{num.strip()}}}{{{denom.strip()}}}"
    
    # Handle square roots
    if '√' in answer:
        if '^2' in answer:
            match = re.search(r'√\((.*?)\)', answer)
            if match:
                return f"\\sqrt{{{match.group(1)}}}"
        return answer.replace('√', r'\sqrt{') + '}'
    
    # Handle points/coordinates
    if answer.startswith('(') and answer.endswith(')'):
        answer = answer.replace('√', r'\sqrt{').replace(')', '})')
        return answer
    
    # Handle remainders
    r_match = re.match(r'^(\d+)\s*R\s*(\d+)$', answer)
    if r_match:
        quotient, remainder = r_match.groups()
        return f"{quotient}R{remainder}"
    
    # Handle basic numbers
    if re.match(r'^-?\d*\.?\d+$', answer):
        return answer
    
    # Handle simple algebraic expressions
    if re.match(r'^-?[a-z0-9\s+\-]+$', answer) and any(c.isalpha() for c in answer):
        return convert_logical_operators(answer.replace(' ', ''))
    
    # If nothing else matched, it's probably not a valid mathematical expression
    return False