import gym
from gym import spaces
import numpy as np
from collections import defaultdict 
from gym.utils import seeding

def fill_matrix(grid_size, n_states, n_actions, prob=0.0, safe_states=[], unsafe_states=[]):
    """construct the transition dynamics as a dictionary of successor states and corresponding probabilities"""
    assert n_states == grid_size**2 

    grid = np.arange(n_states).reshape(grid_size, grid_size)

    # make action map
    action_map = {0: (0, -1), # left
                  1: (0, 1), # right
                  2: (1, 0), # up
                  3: (-1, 0), # down
                  4: (0, 0), # stay
                  5: (-1, -1), # left up
                  6: (-1, 1), # left down
                  7: (-1, 1), # right up
                  8: (1, 1), # right down
                  }

    assert n_actions < len(action_map.keys())

    matrix = np.zeros((n_states, n_states, n_actions))

    # make dictionaries
    probabilities = defaultdict(list)
    successor_states = defaultdict(list)

    for y in range(grid_size):
        for x in range(grid_size):
            for a in range(n_actions):
                state = grid[y][x]
                if (state in safe_states) or (state in unsafe_states):
                    matrix[state, state, a] = 1.0
                    continue

                next_y = int(np.clip(y + action_map[a][0], 0, grid_size-1))
                next_x = int(np.clip(x + action_map[a][1], 0, grid_size-1))
                next_state = grid[next_y, next_x]
                p = 1.0 if state in safe_states else 1.0-prob
                matrix[next_state, state, a] += p

                if p == 1.0:
                    continue

                rand_prob = prob * 1 / (n_actions - 1)
                for rand_a in range(n_actions):
                    if rand_a == a:
                        continue
                    next_y = int(np.clip(y + action_map[rand_a][0], 0, grid_size-1))
                    next_x = int(np.clip(x + action_map[rand_a][1], 0, grid_size-1))
                    next_state = grid[next_y, next_x]
                    matrix[next_state, state, a] += rand_prob

    return matrix

class BridgeCrossing(gym.Env):
    """
    Bridge crossing environment

    The goal is to cross over some lava (unsafe states) by crossing over a safe bridge, from the start state to the safe goal states
    Reward +1 given upon reaching the safe goal states and for each timestep in the safe goal states
    The safe goal states are deterministic states and so form a safe end component
    You can make all other states stochastic with the random_action_probability parameter
    Epsiode length is set to 100 as a default, you can also add entering the safe states as a termination condition

    Input attributes:
        random_action_probability: probability of a random actions being selected
        episode_length: length of the episode until termination
        render_mode: how to render the environment [currently not implemented]

    Other attributes:
        grid_size: size of the grid world
        ncol: number of columns
        nrow: number of rows
        n_states: number of states (grid_size^2)
        n_actions: number of actions
        observation_space: gym spaces object
        action_space: gym spaces object
        reward_fn: the reward function of the environment
        _step_counter: total number of steps in the environment

    """

    metadata = {"render_modes": ["ascii"]}

    def __init__(self, seed=0, random_action_probability=0.04, episode_length=400, render_mode=None):

        self.np_random, _ = seeding.np_random(seed)

        self.grid_size = 20
        self.ncol = self.grid_size
        self.nrow = self.grid_size

        self.n_states = self.grid_size**2
        self.n_actions = 5

        self.random_action_probability=random_action_probability
        self.episode_length = episode_length

        self.observation_space = spaces.Discrete(self.n_states)
        self.action_space = spaces.Discrete(self.n_actions)

        # the start state is in the bottom left corner
        # temporary 
        grid = np.arange(self.n_states).reshape(self.grid_size, self.grid_size)
        self._start_state = grid[-1, 0]

        # the safe goal states form a safe end component
        self._safe_states = list(grid[:7, :].flatten())

        # the unsafe states are still stochastic we just want to avoid them
        self._unsafe_states= list(grid[8:12, :8].flatten()) + list(grid[8:12, -9:].flatten())

        self._terminal_states = self._safe_states + self._unsafe_states

        self.transition_matrix = fill_matrix(self.grid_size, self.n_states, self.n_actions, prob=random_action_probability, safe_states=self._safe_states, unsafe_states=self._unsafe_states)

        self.atomic_predicates = {"start", "goal", "lava"}

        def empty_set():
            return {}

        self.labelling_fn = defaultdict(empty_set) 
        self.labelling_fn[self._start_state.item()] = ({"start"})
        for state in self._safe_states:
            self.labelling_fn[state] = ({"goal"})
        for state in self._unsafe_states:
            self.labelling_fn[state] = ({"lava"})
        
        # get a reward for entering the safe states across the bridge
        self.reward_fn = defaultdict(float)
        for s in self._safe_states:
            self.reward_fn[s] = 1.0

        assert render_mode is None or render_mode in self.metadata["render_modes"]
        self.render_mode = render_mode

        self._step_counter = 0

    def _transition(self, action):
        """sample a next state randomly from the transition matrix"""
        return self.np_random.choice(self.n_states, p=self.transition_matrix[:, self._agent_location, action])

    def _get_labels(self):
        """return the labels for the current state"""
        return self.labelling_fn[self._agent_location]

    def _get_obs(self):
        """return the observation for the current state"""
        return self._agent_location

    def _get_info(self):
        """return the info for the current state"""
        return {"labels": self._get_labels()}

    def _get_reward(self):
        """return the reward for the current state"""
        return self.reward_fn[self._agent_location]

    def _get_terminated(self):
        """check if the termination condition is satisfied"""
        # if you want you can add entering the safe goal states as a termination condition
        # uncomment the following line
        #return True if (self._step_counter >= self.episode_length) or (self._agent_location in self._safe_states) else False
        return True if self._agent_location in self._terminal_states else False
        
    def _get_truncated(self):
        return True if self._step_counter >= self.episode_length else False


    def reset(self, seed=None, options=None):
        """reset the environment and return the start obs"""
        self._agent_location = self._start_state
        observation = self._get_obs()
        info = self._get_info()
        self._step_counter = 0

        if self.render_mode == "ascii":
            self._render_frame()

        return observation, info

    def step(self, action):
        """play a given action in the environment"""
        next_state = self._transition(action)
        self._agent_location = next_state

        # increment step counter
        self._step_counter += 1

        terminated = self._get_terminated()
        truncated = self._get_truncated()
        done = terminated or truncated
        reward = self._get_reward()
        observation = self._get_obs()
        info = self._get_info()
        info["is_truncated"] = truncated
        info["is_terminated"] = terminated

        if self.render_mode == "ascii":
            self._render_frame()
            
        return observation, reward, done, info

    def _render_frame(self):
        grid = np.arange(self.n_states).reshape(self.grid_size, self.grid_size)

        for y in range(self.grid_size):
            row = []
            for x in range(self.grid_size):
                state = grid[y, x]
                if state in self._unsafe_states:
                    row.append("☐")
                elif state in self._safe_states:
                    row.append("G")
                elif state == self._start_state:
                    row.append("S")
                else:
                    row.append("-")
                if state == self._agent_location:
                    row[-1] = "A"

            print(row)
        print(f"Agent in location {self._agent_location}")
