import gym
from gym import spaces
import numpy as np
from collections import defaultdict 
from gym.utils import seeding

def fill_matrix(grid_size, n_states, n_actions, prob=0.0, safe_states=[], wall_states=[]):
    """construct the transition dynamics as a dictionary of successor states and corresponding probabilities"""
    assert n_states == grid_size**2 

    grid = np.arange(n_states).reshape(grid_size, grid_size)

    # make action map
    action_map = {0: (0, -1), # left
                  1: (0, 1), # right
                  2: (1, 0), # up
                  3: (-1, 0), # down
                  4: (0, 0), # stay
                  5: (-1, -1), # left up
                  6: (-1, 1), # left down
                  7: (-1, 1), # right up
                  8: (1, 1), # right down
                  }

    assert n_actions < len(action_map.keys())

    matrix = np.zeros((n_states, n_states, n_actions))

    for y in range(grid_size):
        for x in range(grid_size):
            for a in range(n_actions):
                state = grid[y][x]
                if (state in safe_states):
                    matrix[state, state, a] = 1.0
                    continue
                next_y = int(np.clip(y + action_map[a][0], 0, grid_size-1))
                next_x = int(np.clip(x + action_map[a][1], 0, grid_size-1))
                next_state = grid[next_y, next_x]
                # check if the next state is a wall
                next_state = next_state if next_state not in wall_states else state
                # only the safe/coloured states are deterministic
                p = 1.0 if state in safe_states else 1.0-prob
                matrix[next_state, state, a] += p

                if p == 1.0:
                    continue

                rand_prob = prob * 1 / (n_actions - 1)
                for rand_a in range(n_actions):
                    if rand_a == a:
                        continue
                    next_y = int(np.clip(y + action_map[rand_a][0], 0, grid_size-1))
                    next_x = int(np.clip(x + action_map[rand_a][1], 0, grid_size-1))
                    next_state = grid[next_y, next_x]
                    # check if the next state is a wall
                    next_state = next_state if next_state not in wall_states else state
                    matrix[next_state, state, a] += rand_prob
                    
    return matrix

class ColourBombGridWorld(gym.Env):
    """
    Colour Bomb Gridworld environment

    Input attributes:
        random_action_probability: probability of a random actions being selected
        episode_length: length of the episode until termination
        render_mode: how to render the environment [currently not implemented]

    Other attributes:
        grid_size: size of the grid world
        ncol: number of columns
        nrow: number of rows
        n_states: number of states (grid_size^2)
        n_actions: number of actions
        observation_space: gym spaces object
        action_space: gym spaces object
        reward_fn: the reward function of the environment
        _step_counter: total number of steps in the environment

    """

    metadata = {"render_modes": ["ascii"]}

    def __init__(self, seed=0, random_action_probability=0.0, episode_length=1000, render_mode=None):

        self.np_random, _ = seeding.np_random(seed)

        self.grid_size = 9
        self.ncol = self.grid_size
        self.nrow = self.grid_size

        self.n_states = self.grid_size**2
        # I think we need 5 actions to guarantee the existence of safe end components as we need a stay still action
        self.n_actions = 5

        self.random_action_probability=random_action_probability
        self.episode_length = episode_length

        self.observation_space = spaces.Discrete(self.n_states)
        self.action_space = spaces.Discrete(self.n_actions)

        self._start_state = 74
        self._green_states = [65]
        self._yellow_states = [70, 79]
        self._blue_states = [9, 10, 18, 19]
        self._pink_states = [7, 8, 16, 17]

        # define the walls in the environment
        self._wall_states = [11, 12, 13, 14, 15, 29, 30, 50, 52, 53, 55, 56, 57, 59, 64, 66, 69]

        # all the coloured states are deterministic and therefore each set of coloured states forms a safe end component
        self._safe_states = self._yellow_states + self._blue_states + self._pink_states

        # the unsafe states are bombs we don't want to step on squares that have bombs
        self._bomb_states = [27, 43, 78]

        self.transition_matrix = fill_matrix(self.grid_size, self.n_states, self.n_actions, prob=random_action_probability, safe_states=self._safe_states, wall_states=self._wall_states)

        self.atomic_predicates = {"start", "green", "yellow", "blue", "pink", "bomb"}

        def empty_set():
            return {}

        self.labelling_fn = defaultdict(empty_set) 

        self.labelling_fn[self._start_state] = ({"start"})
        for state in self._green_states:
            self.labelling_fn[state] = ({"green"})
        for state in self._yellow_states:
            self.labelling_fn[state] = ({"yellow"})
        for state in self._blue_states:
            self.labelling_fn[state] = ({"blue"})
        for state in self._pink_states:
            self.labelling_fn[state] = ({"pink"})
        for state in self._bomb_states:
            self.labelling_fn[state] = ({"bomb"})

        # get a reward for entering a coloured state - maybe we need a different reward function since wouldn't an optimal policy go to the nearest coloured state and stay there?
        self.reward_fn = defaultdict(float)
        for s in self._safe_states:
            self.reward_fn[s] = 1.0

        assert render_mode is None or render_mode in self.metadata["render_modes"]
        self.render_mode = render_mode

        self._step_counter = 0

    def _transition(self, action):
        """sample a next state randomly from the transition matrix"""
        return self.np_random.choice(self.n_states, p=self.transition_matrix[:, self._agent_location, action])

    def _get_labels(self):
        """return the labels for the current state"""
        return self.labelling_fn[self._agent_location]

    def _get_obs(self):
        """return the observation for the current state"""
        return self._agent_location

    def _get_info(self):
        """return the info for the current state"""
        return {"labels": self._get_labels()}

    def _get_reward(self):
        """return the reward for the current state"""
        return self.reward_fn[self._agent_location]

    def _get_terminated(self):
        return True if self._agent_location in self._safe_states else False

    def _get_truncated(self):
        return True if self._step_counter >= self.episode_length else False

    def reset(self, seed=None, options=None):
        """reset the environment and return the start obs"""
        self._agent_location = self._start_state
        observation = self._get_obs()
        info = self._get_info()
        self._step_counter = 0

        if self.render_mode == "ascii":
            self._render_frame()

        return observation, info

    def step(self, action):
        """play a given action in the environment"""
        next_state = self._transition(action)
        self._agent_location = next_state

        # increment step counter
        self._step_counter += 1

        terminated = self._get_terminated()
        truncated=self._get_truncated()
        done = terminated or truncated
        reward = self._get_reward()
        observation = self._get_obs()
        info = self._get_info()
        info["is_truncated"] = truncated
        info["is_terminated"] = terminated

        if self.render_mode == "ascii":
            self._render_frame()
            
        return observation, reward, done, info

    def _render_frame(self):

        grid = np.arange(self.n_states).reshape(self.grid_size, self.grid_size)

        for y in range(self.grid_size):
            row = []
            for x in range(self.grid_size):
                state = grid[y, x]
                if state in self._wall_states:
                    row.append("☐")
                elif state in self._green_states:
                    row.append("G")
                elif state in self._yellow_states:
                    row.append("Y")
                elif state in self._blue_states:
                    row.append("B")
                elif state in self._pink_states:
                    row.append("P")
                elif state in self._bomb_states:
                    row.append("O")
                elif state == self._start_state:
                    row.append("S")
                else:
                    row.append("-")
                if state == self._agent_location:
                    row[-1] = "A"

            print(row)
        print(f"Agent in location {self._agent_location}")
        return None