

# ===================
# Part 1: Importing Libraries
# ===================
import math
import matplotlib.pyplot as plt
from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Function to create polygons under the graph
def polygon_under_graph(x, y):
    """
    Construct the vertex list which defines the polygon filling the space under
    the (x, y) line graph. This assumes x is in ascending order.
    """
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]

# New dataset for energy consumption
time_hours = np.linspace(0.0, 24.0, 31)
building_types = ["Residential", "Commercial", "Industrial"]
energy_consumption = lambda building, hours: (np.sin(hours / 6) + 1) * np.random.uniform(0.8, 1.2)  # Simulated energy consumption data

# Encode building types as integers for plotting
building_type_indices = [0, 1, 2]

# Create polygons to represent data
verts = [
    polygon_under_graph(time_hours, energy_consumption(building, time_hours))
    for building in building_types
]

# Labels
xlabel = "Time (hours)"
ylabel = "Building Type"
zlabel = "Energy Consumption (kWh)"
title = "Energy Consumption for Different Building Types Over a Day"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
fig = plt.figure(figsize=(10, 8))
ax = fig.add_subplot(projection="3d")
facecolors = plt.cm.viridis(np.linspace(0, 1, len(verts)))

poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax.add_collection3d(poly, zs=building_type_indices, zdir="y")

ax.set(
    xlim=(0, 24),
    ylim=(min(building_type_indices), max(building_type_indices) + 1),
    zlim=(0, 3),
    xlabel=xlabel,
    ylabel=ylabel,
    zlabel=zlabel,
    title=title
)

ax.set_yticks(building_type_indices)
ax.set_yticklabels(building_types)
ax.grid(True)

# Additional styling adjustments
ax.view_init(elev=20., azim=-35)

plt.tight_layout()
plt.savefig("3d_62.pdf", bbox_inches="tight")
