
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)

# Function to create polygon under graph
def polygon_under_graph(x, y):
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]

# Data for bar chart
years = [2010, 2011, 2012, 2013, 2014, 2015, 2016, 2017, 2018, 2019]
GDP_Growth = [2.5, 1.8, 2.2, 2.9, 2.5, 3.1, 2.7, 2.9, 2.3, 2.5]
Unemployment_Rate = [9.6, 9.0, 8.1, 7.4, 6.2, 5.3, 4.9, 4.4, 4.0, 3.7]

# Data for distribution graph
x = np.linspace(0.0, 10.0, 31)
economic_factors = range(1, 4)
exp = np.exp
verts = [polygon_under_graph(x, exp(-0.5 * (x - t) ** 2)) for t in economic_factors]

# Labels
xlabels = ["Year", "Time Since Event (Years)"]
ylabels = ["Economic Indicator", "Factor Level"]
zlabels = ["Percentage (%)", "Impact Rate"]
legend_labels = ["GDP Growth", "Unemployment Rate"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Custom color palette
facecolors = plt.colormaps["viridis"](np.linspace(0, 1, len(verts)))

# Initialize figure and axes
fig = plt.figure(figsize=(12, 7))
ax1 = fig.add_subplot(121, projection="3d")  # 3D bar chart
ax2 = fig.add_subplot(122, projection="3d")  # 3D distribution graph

# Bar chart data
ax1.bar(years, GDP_Growth, zs=0, zdir="y", color="#ff7f0e", alpha=0.8)
ax1.bar(years, Unemployment_Rate, zs=1, zdir="y", color="#1f77b4", alpha=0.8)

# Set labels and ticks for bar chart
ax1.set_xlabel(xlabels[0])
ax1.set_ylabel(ylabels[0])
ax1.set_zlabel(zlabels[0])
ax1.set_yticks([0, 1])
ax1.set_yticklabels(legend_labels)
ax1.set_title("Economic Indicators (2010-2019)")
ax1.tick_params(axis='y', pad=-5)

# Add polygons to the plot
poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax2.add_collection3d(poly, zs=economic_factors, zdir="y")

# Set labels and limits for distribution graph
ax2.set(xlim=(0, 10), ylim=(1, 4), zlim=(0, 1), xlabel=xlabels[1], ylabel=ylabels[1], zlabel=zlabels[1])
ax2.set_yticks([1, 2, 3])
ax2.set_title("Economic Factor Impact Over Time")

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the figure
plt.tight_layout()
ax1.set_box_aspect(aspect=None, zoom=0.9)
ax2.set_box_aspect(aspect=None, zoom=0.8)
plt.savefig("3d_64.pdf", bbox_inches="tight")
