
# ===================
# Part 1: Importing Libraries
# ===================
import math
import matplotlib.pyplot as plt
from matplotlib.collections import PolyCollection

# ===================
# Part 2: Data Preparation
# ===================
# Set a random seed for reproducibility
import numpy as np

np.random.seed(42)

def polygon_under_graph(x, y):
    """
    Construct the vertex list which defines the polygon filling the space under
    the (x, y) line graph. This assumes x is in ascending order.
    """
    return [(x[0], 0.0), *zip(x, y), (x[-1], 0.0)]

# Define the x-axis data (Years)
x = np.arange(1, 11)  # Years: 1 to 10
strategies = range(1, 4)  # Three portfolio strategies

# New function to simulate portfolio growth over years
def portfolio_growth(x, strategy):
    # Simulate exponential portfolio growth for different strategies
    base_growth_rate = [0.05, 0.1, 0.2]  # Conservative, Balanced, Aggressive
    return (1 + base_growth_rate[strategy-1]) ** x + np.random.normal(0, 0.5, len(x))

# Generate vertices for polygons
verts = [polygon_under_graph(x, portfolio_growth(x, strategy)) for strategy in strategies]

# Extracted variables
x_label = "Years"
y_label = "Portfolio Strategy"
z_label = "Total Value (Billions)"
title = "Growth of Stock Portfolios Over 10 Years"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure and 3D axis
yticks = [1, 2, 3]
xlim = (1, 10)
ylim = (1, 4)
zlim = (0, 10)
aspect_ratio = [2, 1, 1]
view_elev = 30
view_azim = 210

fig = plt.figure(figsize=(12, 9))
ax = fig.add_subplot(projection="3d")

# Define the face colors
facecolors = plt.get_cmap("autumn")(np.linspace(0, 1, len(verts)))

# Create a PolyCollection object
poly = PolyCollection(verts, facecolors=facecolors, alpha=0.7)
ax.add_collection3d(poly, zs=strategies, zdir="y")

# Set the axis labels and limits
ax.set(xlim=xlim, ylim=ylim, zlim=zlim, xlabel=x_label, ylabel=y_label, zlabel=z_label)

# Add ticks and title
ax.set_yticks(yticks)
ax.set_title(title, pad=20)

# Adjust the aspect ratio and view angle
ax.set_box_aspect(aspect_ratio)
ax.view_init(elev=view_elev, azim=view_azim)

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the figure
plt.tight_layout()
plt.savefig("3d_80.pdf", bbox_inches="tight")
