
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Setting a seed for reproducibility

# Generating random data to represent the time to execute plays (in minutes) for different types of players
data_forwards = np.random.normal(70, 10, 100)  # Forwards
data_midfielders = np.random.normal(60, 8, 100)  # Midfielders
data_defenders = np.random.normal(80, 12, 100)  # Defenders
data_goalkeepers = np.random.normal(50, 7, 100)  # Goalkeepers

# Packing the data into a list
data = [data_forwards, data_midfielders, data_defenders, data_goalkeepers]
legend_labels = ["Forwards", "Midfielders", "Defenders", "Goalkeepers"]
line_label = "Median Play Duration"
ylabel = "Play Duration (Minutes)"
xlabel = "Type of Player"
title = "Comparison of Play Durations by Player Type"
label = "Median Play Duration"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a box plot
fig, ax1 = plt.subplots(figsize=(10, 6))
bp = ax1.boxplot(
    data, patch_artist=True, notch=False, showfliers=False, positions=[1, 2, 3, 4]
)

# Customizing the boxplot colors
colors = ["#4b9cd3", "#66c2a5", "#fc8d62", "#8da0cb"]
for patch, color in zip(bp["boxes"], colors):
    patch.set_facecolor(color)
for median in bp["medians"]:
    median.set(color="black", zorder=2)

# Extracting medians for the line graph
medians = [np.median(d) for d in data]

# Creating the line graph on the same axes
ax1.plot(
    [1, 2, 3, 4],
    medians,
    "-o",
    color="black",
    label=label,
    ms=10,
    markerfacecolor="#e41a1c",
)

# Setting legend for the boxplot
legend_patches = [
    mpatches.Patch(color=color, label=label)
    for color, label in zip(colors, legend_labels)
]
ax1.legend(
    handles=legend_patches + [mpatches.Patch(color="#e41a1c", label=line_label)],
    loc="upper left",
)

# Setting labels for the x-axis
ax1.set_xticklabels(legend_labels)

# Setting the y-axis label and title
ax1.set_ylabel(ylabel)
ax1.set_xlabel(xlabel)
ax1.set_title(title)

# Setting y-axis limits and adding grid lines
ax1.set_ylim(30, 110)
ax1.yaxis.grid(True, which="major", linestyle="--", color="grey", alpha=0.5)

# Removing top and right spines for aesthetics
ax1.spines["top"].set_visible(False)
ax1.spines["right"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_109.pdf", bbox_inches="tight")
