
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.patches as mpatches

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)  # Setting a seed for reproducibility

# Generating random data to represent the annual revenue (in millions of dollars) for different tech companies
data_apple = np.random.normal(275, 20, 100)  # Apple
data_microsoft = np.random.normal(160, 15, 100)  # Microsoft
data_google = np.random.normal(182, 18, 100)  # Google
data_amazon = np.random.normal(232, 22, 100)  # Amazon

# Packing the data into a list
data = [data_apple, data_microsoft, data_google, data_amazon]
legend_labels = ["Apple", "Microsoft", "Google", "Amazon"]
line_label = "Median Annual Revenue"
ylabel = "Annual Revenue (in millions)"
xlabel = "Tech Companies"
label = "Median Annual Revenue"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Creating a box plot
fig, ax1 = plt.subplots(figsize=(10, 6))
bp = ax1.boxplot(
    data, patch_artist=True, notch=False, showfliers=False, positions=[1, 2, 3, 4]
)

# Customizing the boxplot colors
colors = ["#8c510a", "#d8b365", "#c7eae5", "#5ab4ac"]  # Earthy, historical tones
for patch, color in zip(bp["boxes"], colors):
    patch.set_facecolor(color)
for median in bp["medians"]:
    median.set(color="black", zorder=2)

# Extracting medians for the line graph
medians = [np.median(d) for d in data]

# Creating the line graph on the same axes
ax1.plot(
    [1, 2, 3, 4],
    medians,
    "-o",
    color="black",
    label=label,
    ms=10,
    markerfacecolor="#e66101",
)

# Setting legend for the boxplot
legend_patches = [
    mpatches.Patch(color=color, label=label)
    for color, label in zip(colors, legend_labels)
]
ax1.legend(
    handles=legend_patches + [mpatches.Patch(color="#e66101", label=line_label)],
    loc="upper right",
)

# Setting labels for the x-axis
ax1.set_xticklabels(legend_labels)

# Setting the y-axis label
ax1.set_ylabel(ylabel)
ax1.set_xlabel(xlabel)

# Setting y-axis limits and adding grid lines
ax1.set_ylim(100, 350)
ax1.yaxis.grid(True, which="major", linestyle="--", color="grey", alpha=0.5)

# Removing top and right spines for aesthetics
ax1.spines["top"].set_visible(False)
ax1.spines["right"].set_visible(False)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_111.pdf", bbox_inches="tight")
