

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import norm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)
# New Sample data for energy production
renewable_energy = np.random.normal(loc=50, scale=10, size=500)
nonrenewable_energy = np.random.normal(loc=100, scale=15, size=500)

# Labels and Plot Types
ax1_label = "Renewable Energy"
ax2_label = "Non-Renewable Energy"
ax1_text = r"Average Renewable"
ax2_text = r"Average Non-Renewable"

# Axes Limits and Labels
xlabel_value = "Energy Production (MWh)"
ylabel_value = "Frequency"
xticks_values = [20, 40, 60, 80, 100, 120, 140]
xlim_values = [15, 145]
title='Comparison of Energy Production Between Renewable and Non-Renewable Sources'

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axis
fig, ax = plt.subplots(
    figsize=(10, 8)
)

# Plot histograms
ax.hist(renewable_energy, bins=15, density=True, alpha=0.7, color="#69b3a2", edgecolor='black', label=ax1_label)
ax.hist(nonrenewable_energy, bins=15, density=True, alpha=0.7, color="#404080", edgecolor='black', label=ax2_label)

# Plot normal distributions
xmin, xmax = ax.get_xlim()
x = np.linspace(xmin, xmax, 100)
p_renewable = norm.pdf(x, np.mean(renewable_energy), np.std(renewable_energy))
p_nonrenewable = norm.pdf(x, np.mean(nonrenewable_energy), np.std(nonrenewable_energy))
ax.plot(x, p_renewable, color="#1aaf54", linestyle='-', linewidth=3)
ax.plot(x, p_nonrenewable, color="#a52a2a", linestyle='--', linewidth=3)

# Add dashed lines at mean
ax.axvline(np.mean(renewable_energy), color="black", linestyle="dashed")
ax.axvline(np.mean(nonrenewable_energy), color="black", linestyle="dashed")

# Add text labels for dashed lines
ax.text(
    np.mean(renewable_energy) + 1,
    ax.get_ylim()[1] - 0.02,
    ax1_text,
    ha="center",
    va="top",
    fontsize=14,
)
ax.text(
    np.mean(nonrenewable_energy) + 1,
    ax.get_ylim()[1] - 0.04,
    ax2_text,
    ha="center",
    va="top",
    fontsize=14,
)

# Set labels and title
ax.set_xlabel(xlabel_value)
ax.set_ylabel(ylabel_value)
ax.set_title(title)
ax.set_xticks(xticks_values)
ax.set_xlim(xlim_values)
ax.grid(True, linestyle='--', alpha=0.7)

# Add legend
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout()
plt.savefig("CB_31.pdf", bbox_inches="tight")
