

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import norm

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Simulating endorsement levels for Empiricism and Rationalism
endorsement_empiricism = np.random.normal(loc=7, scale=1.5, size=500)
endorsement_rationalism = np.random.normal(loc=5, scale=1.2, size=500)

# Labels and Plot Types
ax1_label = "Empiricism"
ax2_label = "Rationalism"
ax1_text = r"$\mathrm{Empiricism}$"
ax2_text = r"$\mathrm{Rationalism}$"

# Axes Limits and Labels
xlabel_value = "Endorsement Level"
ylabel_value = "Density"
xticks_values = [0, 2, 4, 6, 8, 10]
xlim_values = [-1, 11]
title="Distribution of Philosophical Endorsements"
# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create figure and axis
fig, ax = plt.subplots(
    figsize=(10, 8)
)

# Plot histograms
ax.hist(endorsement_empiricism, bins=15, density=True, alpha=0.75, color="#FF6347", label=ax1_label)
ax.hist(endorsement_rationalism, bins=15, density=True, alpha=0.75, color="#4682B4", label=ax2_label)

# Plot normal distributions
xmin, xmax = ax.get_xlim()
x = np.linspace(xmin, xmax, 100)
p_empiricism = norm.pdf(x, np.mean(endorsement_empiricism), np.std(endorsement_empiricism))
p_rationalism = norm.pdf(x, np.mean(endorsement_rationalism), np.std(endorsement_rationalism))
ax.plot(x, p_empiricism, color="#FF4500", linewidth=3, linestyle='solid')
ax.plot(x, p_rationalism, color="#1E90FF", linewidth=3, linestyle='solid')

# Add dashed lines at mean
ax.axvline(np.mean(endorsement_empiricism), color="black", linestyle="dashed")
ax.axvline(np.mean(endorsement_rationalism), color="black", linestyle="dashed")

# Add text labels for dashed lines
ax.text(
    np.mean(endorsement_empiricism) + 0.1,
    ax.get_ylim()[1] - 0.01,
    ax1_text,
    ha="center",
    va="top",
    fontsize=18,
)
ax.text(
    np.mean(endorsement_rationalism) + 0.1,
    ax.get_ylim()[1] - 0.005,
    ax2_text,
    ha="center",
    va="top",
    fontsize=18,
)

# Set labels and title
ax.set_xlabel(xlabel_value, fontsize=14, labelpad=15)
ax.set_ylabel(ylabel_value, fontsize=14, labelpad=15)
ax.set_xticks(xticks_values)
ax.set_xlim(xlim_values)
ax.set_title(title, fontsize=16, pad=20)

# Add legend
ax.legend()

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and save plot
plt.tight_layout()
plt.savefig("CB_32.pdf", bbox_inches="tight")

