

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Define parameters for three cities
distributions = {
    "CityA": {"mean": [30, 70], "cov": [[10, 15], [15, 40]]},
    "CityB": {"mean": [20, 60], "cov": [[8, 12], [12, 35]]},
    "CityC": {"mean": [25, 65], "cov": [[9, 14], [14, 38]]},
}
keys = ["CityA", "CityB", "CityC"]
# Generate samples
samples = {
    city: np.random.multivariate_normal(dist["mean"], dist["cov"], 100)
    for city, dist in distributions.items()
}

xlabel = "Temperature (°C)"
ylabel = "Humidity (%)"
title = "Weather Data by City"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the plotting grid
fig = plt.figure(figsize=(10, 8))
grid = plt.GridSpec(4, 4, hspace=0, wspace=0)

# Main scatter plot
main_ax = fig.add_subplot(grid[1:, :-1])
colors = {keys[0]: "teal", keys[1]: "orange", keys[2]: "purple"}
markers = {keys[0]: "o", keys[1]: "^", keys[2]: "s"}
for city, color in colors.items():
    subset = samples[city]
    main_ax.scatter(subset[:, 0], subset[:, 1], c=color, label=city, marker=markers[city], alpha=0.6)
main_ax.set_xlabel(xlabel)
main_ax.set_ylabel(ylabel)

# Top density plot
top_ax = fig.add_subplot(grid[0, :-1], sharex=main_ax)
all_samples = np.concatenate([samples[city] for city in samples], axis=0)
x_min, x_max = all_samples[:, 0].min(), all_samples[:, 0].max()
xs = np.linspace(x_min, x_max, 200)
for city, color in colors.items():
    density = gaussian_kde(samples[city][:, 0])
    top_ax.plot(xs, density(xs), linestyle='--', alpha=0.6, color=color)
    top_ax.fill_between(xs, density(xs), alpha=0.4, color=color)

# Right density plot
right_ax = fig.add_subplot(grid[1:, -1], sharey=main_ax)
y_min, y_max = all_samples[:, 1].min(), all_samples[:, 1].max()
ys = np.linspace(y_min, y_max, 200)
for city, color in colors.items():
    density = gaussian_kde(samples[city][:, 1])
    right_ax.plot(density(ys), ys, linestyle='--', alpha=0.6, color=color)
    right_ax.fill_betweenx(ys, density(ys), alpha=0.4, color=color)

# Hide the spines
top_ax.spines["top"].set_visible(False)
top_ax.spines["right"].set_visible(False)
top_ax.spines["left"].set_visible(False)
right_ax.spines["top"].set_visible(False)
right_ax.spines["right"].set_visible(False)
right_ax.spines["bottom"].set_visible(False)

# Remove the labels from the top and right axes
top_ax.tick_params(axis="x", which="both", top=False, bottom=False, labelbottom=False)
right_ax.tick_params(axis="y", which="both", left=False, right=False, labelleft=False)

# Remove all ticks from the right axis
top_ax.set_yticks([])
right_ax.set_xticks([])

main_ax.legend(title=title)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_48.pdf", bbox_inches="tight")

