

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from scipy.stats import gaussian_kde

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Sample data for demonstration purposes
temperature_data = np.random.normal(loc=20, scale=5, size=1000)  # Daily temperatures in Celsius
humidity_data = np.random.uniform(low=30, high=90, size=1000)    # Daily humidity percentage

# Compute KDE for both datasets
kde_temperature = gaussian_kde(temperature_data)
kde_humidity = gaussian_kde(humidity_data)

# Create an array of values for plotting KDE
x_eval = np.linspace(
    min(np.concatenate([temperature_data, humidity_data])),
    max(np.concatenate([temperature_data, humidity_data])),
    1000,
)

labels = ["Temperature Data", "Humidity Data", "Temperature KDE", "Humidity KDE"]
p_text = "Correlation coefficient = 0.23"
x_label = "Measurement Values"
y_label = "Density"
title = "Comparison of Daily Temperatures and Humidity Levels"
suptitle = "Weather Data Visualization using KDE and Histogram"
legend_labels = ["Humidity Data", "Temperature Data"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Set the figure size to match the original image's dimensions
plt.figure(figsize=(10, 6))

# Plot histograms
plt.hist(
    humidity_data, density=True, bins=30, color="#87CEEB", alpha=0.7, label=legend_labels[0]
)
plt.hist(
    temperature_data, density=True, bins=30, color="#FFB6C1", alpha=0.7, label=legend_labels[1]
)

# Plot KDEs
plt.plot(
    x_eval,
    kde_humidity(x_eval),
    linestyle="dotted",
    color="blue",
    label=labels[3],
    linewidth=2
)
plt.plot(
    x_eval,
    kde_temperature(x_eval),
    linestyle="dotted",
    color="red",
    label=labels[2],
    linewidth=2
)

# Add legend
plt.legend()

# Add correlation coefficient text
plt.text(30, 0.04, p_text, fontsize=10, bbox=dict(facecolor='white', alpha=0.5))

# Set labels and title
plt.xlabel(x_label)
plt.ylabel(y_label)
plt.title(title)
plt.suptitle(suptitle, fontsize=12)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("CB_66.pdf", bbox_inches="tight")

