

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate data for the religion domain
weeks = np.arange(1, 101)
weekly_attendance = np.random.normal(100, 20, 100)  # Normal distribution of attendance around 100 with std deviation of 20
perceived_support = weekly_attendance * 0.8 + np.random.randn(100) * 10  # Simulate perceived support with some noise
trend_x = np.linspace(50, 150, 5)
trend_y = trend_x * 0.8

# Labels and legends
scatter_name = "Weekly Attendance vs. Support"
line_name = "Support Trend Line"
xlabel = "Weekly Religious Attendance"
ylabel = "Perceived Community Support"
title = "Perceived Community Support vs. Religious Attendance"
supertitle = "Religious Community: Support Analysis"
legend_location = "upper left"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Start with a square Figure.
fig = plt.figure(figsize=(6, 6))
gs = fig.add_gridspec(
    2,
    1,
    height_ratios=(1, 4),
    left=0.1,
    right=0.9,
    bottom=0.1,
    top=0.9,
    wspace=0.0,
    hspace=0.0,
)
# Create the Axes.
ax = fig.add_subplot(gs[1, 0])
ax_histx = fig.add_subplot(gs[0, 0], sharex=ax)

# Draw the scatter plot and marginals.
# no labels
ax_histx.tick_params(axis="x", labelbottom=False)

# Scatter plot:
ax.scatter(weekly_attendance, perceived_support, color="#4CAF50", edgecolor="white", s=50, label=scatter_name)

# Add a trend line on ax
ax.plot(
    trend_x,
    trend_y,
    color="#2196F3",
    linewidth=2,
    marker="o",
    markersize=6,
    label=line_name,
)

# Draw a diagonal reference line
ax.plot([50, 150], [40, 120], color="grey", linestyle="--")

# Determine nice limits by hand
binwidth = 10
xymax = max(np.max(np.abs(weekly_attendance)), np.max(np.abs(perceived_support)))
lim = (int(xymax / binwidth) + 1) * binwidth
bins = np.arange(50, lim + binwidth, binwidth)
ax_histx.hist(weekly_attendance, bins=bins, color="#A5D6A7", edgecolor="black")

# Remove y-axis labels for histogram
ax_histx.set_yticks([])

# Remove the top, left, and right spines
ax_histx.spines["top"].set_visible(False)
ax_histx.spines["left"].set_visible(False)
ax_histx.spines["right"].set_visible(False)

# Set titles and labels
fig.suptitle(supertitle, fontsize=14)
ax.set_title(title, fontsize=12)
ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)

# Set ticks and limits for the scatter plot
ax.set_xticks([50, 75, 100, 125, 150])
ax.set_yticks([40, 60, 80, 100, 120])
ax.set_xlim(40, 160)
ax.set_ylim(30, 130)

# Add legend
ax.legend(loc=legend_location)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("CB_90.pdf", bbox_inches="tight")
