
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(42)

# Define the bubble sizes and colors for each task
bubble_sizes = {100: 300, 75: 250, 50: 200, 25: 150, 0: 50}

label2idx = {"Temperature": 0, "Rainfall": 1, "Humidity": 2, "Wind Speed": 3}
idx2label = {v: k for k, v in label2idx.items()}

data = {
    "NA": {
        "Temperature": [100, 75, 50, 25],
        "Rainfall": [50, 75, 25, 100],
        "Humidity": [25, 50, 75, 100],
        "Wind Speed": [75, 25, 50, 0],
    },
    "EU": {
        "Temperature": [50, 25, 75, 0],
        "Rainfall": [75, 50, 25, 0],
        "Humidity": [100, 75, 50, 25],
        "Wind Speed": [25, 100, 75, 50],
    },
    "AS": {
        "Temperature": [75, 50, 25, 100],
        "Rainfall": [25, 75, 100, 50],
        "Humidity": [50, 100, 25, 75],
        "Wind Speed": [100, 25, 75, 50],
    },
    "AF": {
        "Temperature": [25, 75, 50, 100],
        "Rainfall": [100, 50, 75, 25],
        "Humidity": [75, 25, 100, 50],
        "Wind Speed": [50, 75, 100, 25],
    },
}

title = "Climate Metrics"
suptitle="Regional Climate Metrics"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the subplots
fig, axs = plt.subplots(1, 4, figsize=(12, 4), sharey=True)
colors = {0: "#ff9999", 1: "#66b3ff", 2: "#99ff99", 3: "#ffcc99"}

# Loop through each subplot and plot the data
for i, (title, metrics) in enumerate(data.items()):
    ax = axs[i % 4]
    ax.set_title(title)

    for metric, values in metrics.items():
        ax.scatter(
            x=range(len(values)),
            y=[label2idx[metric]] * len(values),
            s=[bubble_sizes[_] for _ in values],
            c=colors[label2idx[metric]],
            label=metric,
            alpha=0.75,
            edgecolors='w',
            linewidth=0.5,
        )
        for j, v in enumerate(values):
            ax.text(j, label2idx[metric], str(v), ha="center", va="center", fontsize=8)
    ax.set_xticks(range(len(values)))
    ax.set_xticklabels([chr(65+i) for i in range(len(values))])
    ax.set_xlim(-0.5, len(values) - 0.5)
    ax.set_ylim(-0.5, len(label2idx) - 0.5)
    ax.set_yticks(range(len(label2idx)))
    ax.set_yticklabels([idx2label[i] for i in range(len(label2idx))])
    ax.set_xlabel("Client")

fig.suptitle(suptitle, fontsize=14)

# Add the legend
handles, labels = axs[0].get_legend_handles_labels()
fig.legend(
    handles,
    labels,
    loc="lower center",
    ncol=4,
    markerscale=0.75,
    fontsize="small",
    bbox_to_anchor=(0.5, -0.1),
    title=title,
    title_fontsize="medium",
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the plot
plt.tight_layout()
plt.savefig("HR_26.pdf", bbox_inches="tight")
