

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np  # Import numpy library in Part 2 for data preparation
np.random.seed(0)

# Define the bubble sizes and colors for each category
bubble_sizes = {100: 500, 80: 450, 60: 400, 40: 300, 20: 150, 0: 20}
label2idx = {"Prayer Frequency": 0, "Service Attendance": 1, "Scripture Reading": 2, "Charitable Acts": 3}
idx2label = {v: k for k, v in label2idx.items()}

data = {
    "Group A": {
        "Prayer Frequency": [60, 60, 60, 60],
        "Service Attendance": [60, 60, 60, 60],
        "Scripture Reading": [60, 60, 60, 60],
        "Charitable Acts": [60, 60, 60, 60],
    },
    "Group B": {
        "Prayer Frequency": [0, 0, 0, 80],
        "Service Attendance": [0, 0, 80, 0],
        "Scripture Reading": [0, 80, 0, 0],
        "Charitable Acts": [80, 0, 0, 0],
    },
    "Group C": {
        "Prayer Frequency": [20, 40, 60, 100],
        "Service Attendance": [20, 40, 60, 100],
        "Scripture Reading": [20, 40, 60, 100],
        "Charitable Acts": [20, 40, 60, 100],
    },
    "Group D": {
        "Prayer Frequency": [0, 0, 0, 80],
        "Service Attendance": [0, 0, 80, 0],
        "Scripture Reading": [0, 40, 0, 0],
        "Charitable Acts": [20, 0, 0, 0],
    },
}

title = "Religious Practices Assessment"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Define color scheme
colors = {0: "#ff9999", 1: "#66b3ff", 2: "#99ff99", 3: "#ffcc99"}

# Create the subplots
fig, axs = plt.subplots(1, 4, figsize=(10, 3), sharey=True)

# Loop through each subplot and plot the data
for i, (title, tasks) in enumerate(data.items()):
    ax = axs[i % 4]
    ax.set_title(title)

    for task, values in tasks.items():
        ax.scatter(
            x=range(len(values)),
            y=[label2idx[task]] * len(values),
            s=[bubble_sizes[_] for _ in values],
            c=colors[label2idx[task]],
            label=task,
            alpha=0.6,
            edgecolors="w",
            linewidth=0.5,
            marker="o" if task == "Prayer Frequency" else "s" if task == "Service Attendance" else "D" if task == "Scripture Reading" else "P"
        )
        for j, v in enumerate(values):
            ax.text(j, label2idx[task], str(v), ha="center", va="center", fontsize=8)
    ax.set_xticks(range(len(values)))
    ax.set_xticklabels([i for i in range(len(values))])
    ax.set_xlim(-0.5, len(values) - 0.5)
    ax.set_ylim(-0.5, len(label2idx) - 0.5)
    ax.set_yticks(range(len(label2idx)))
    ax.set_yticklabels([idx2label[idx] for idx in range(len(label2idx))])
    ax.set_xlabel("Participant")

# Add the legend
handles, labels = axs[0].get_legend_handles_labels()

# set the title 'Religious Practices Assessment' of legend to the left of the legend
fig.legend(
    handles,
    labels,
    loc="lower center",
    ncol=4,
    markerscale=0.5,
    fontsize="small",
    bbox_to_anchor=(0.5, -0.1),
    title=title,
    title_fontsize="small",
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust the layout and save the plot
plt.tight_layout()
plt.savefig("HR_28.pdf", bbox_inches="tight")

