
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.cm as cm
import matplotlib.colors as colors

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(1)

# Function to draw ellipses with varying radii and colors
xlim = [0, 50]
ylim = [-10, 15]
xlabel = "Volatility (%)"
ylabel = "Average Return (%)"
num_ellipses = 12
a_start = 5
b_start = -5
a_end = 25
b_end = 10
title = "Stock Performance Analysis"
color_labels = ["Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================

def draw_colored_ellipses(num_ellipses, a_start, b_start, a_end, b_end):
    t = np.linspace(0, 2 * np.pi, 100)
    fig, ax = plt.subplots(figsize=(8, 6))

    # Create a colormap
    cmap = cm.get_cmap("coolwarm", num_ellipses)
    norm = colors.BoundaryNorm(np.arange(num_ellipses + 1), cmap.N)

    for i in range(num_ellipses):
        # Interpolate the semi-major and semi-minor axes
        a = np.linspace(a_start, a_end, num_ellipses)[i]
        b = np.linspace(b_start, b_end, num_ellipses)[i]

        # Parametric equations for the ellipse
        x = a * np.cos(t)
        y = b * np.sin(t)

        # Use a colormap to determine the color
        color = cmap(i)
        ax.set_xlim(xlim)
        ax.set_ylim(ylim)
        ax.plot(x, y, linestyle='--', linewidth=1.5, color=color)
        ax.set_xlabel(xlabel)
        ax.set_ylabel(ylabel)

    # Add some annotations
    ax.annotate("High Volatility", xy=(15, 5), xytext=(20, 10),
                arrowprops=dict(facecolor='black', shrink=0.05))

    # Add a grid
    ax.grid(True, linestyle='--', linewidth=0.5)
    ax.set_aspect("equal")

    # Add Title
    ax.set_title(title)

    # Create a mappable object for the colorbar
    sm = cm.ScalarMappable(cmap=cmap, norm=norm)
    sm.set_array([])

    # Add the colorbar
    cbar = plt.colorbar(sm, ticks=np.arange(0.5, num_ellipses, 1), label="Month", ax=ax)
    cbar.set_ticklabels(color_labels)  # set tick labels to specified times
    cbar.ax.tick_params(length=0)


# Show the plot
draw_colored_ellipses(
    num_ellipses=num_ellipses,
    a_start=a_start,
    b_start=b_start,
    a_end=a_end,
    b_end=b_end,
)
plt.tight_layout()

# ===================
# Part 4: Saving Output
# ===================
plt.savefig("HR_29.pdf", bbox_inches="tight")
