
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
from matplotlib.patches import Ellipse

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np
np.random.seed(0)

# Generating data representing vehicle speed over time with error ellipses
time = np.linspace(0, 24, 400)  # Time from 0 to 24 hours
average_speed = 50 + 10 * np.sin(time / 24 * 2 * np.pi)  # Sinusoidal function for speed trend
error_magnitude = 5 + 2 * np.sin(time / 24 * 2 * np.pi)  # Variable error magnitude

# Measurement points with some offset
measurement_time = np.linspace(1, 23, 20)
measurement_speed = 50 + 10 * np.sin(measurement_time / 24 * 2 * np.pi) - 1

# Labels and Plot Types
label_trend = "Average Vehicle Speed"
label_error = "± 1-σ of Measurement"
label_measurement = "Individual Measurements"
ylim_values = [30, 70]
ellipse_sizes = 3 * np.ones_like(measurement_time)

# Axes Limits and Labels
xlabel_value = "Time (hours)"
ylabel_value = "Speed (km/h)"
title = "Vehicle Speed Over 24 Hours with Measurement Error"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 6))
plt.plot(time, average_speed, label=label_trend, color="#4CAF50", linestyle="--", linewidth=2)

for i, (px, py, size) in enumerate(zip(measurement_time, measurement_speed, ellipse_sizes)):
    if i == 0:  # Only for the first ellipse add label
        ellipse = Ellipse(
            (px, py),
            width=size,
            height=size / 2,
            edgecolor="#388E3C",
            facecolor="none",
            label=label_error,
        )
    else:
        ellipse = Ellipse(
            (px, py), width=size, height=size / 2, edgecolor="#388E3C", facecolor="none"
        )
    plt.gca().add_patch(ellipse)

plt.scatter(measurement_time, measurement_speed, color="#388E3C", label=label_measurement, zorder=5, marker='o')
plt.xlabel(xlabel_value)
plt.ylabel(ylabel_value)
plt.title(title)
plt.ylim(ylim_values)
plt.grid(True, color="white")
plt.gca().set_facecolor("#f4f4f4")
plt.gca().set_axisbelow(True)

plt.legend(facecolor="#f4f4f4")
for spine in plt.gca().spines.values():
    spine.set_visible(False)
plt.gca().tick_params(axis="both", length=0)  # Hide tick marks

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_35.pdf", bbox_inches="tight")
