

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Generate new psychology-related data
months = np.linspace(0, 12, 400)
stress_levels = 80 * np.exp(-months / 6) + 10 * np.sin(np.pi * months / 6)

# Generate dense points for error representation
points_months = np.linspace(1, 11, 20)
points_stress_levels = 80 * np.exp(-points_months / 6) + 10 * np.sin(np.pi * points_months / 6)
error_size = np.linspace(1, 5, len(points_months))

# Text labels and plot types
label_1 = "Stress Level Trend"
label_2 = "± 3-σ Variability"
label_3 = "Mean Stress Levels"
ylim_values = [0, 100]

xlabel_value = "Time (months)"
ylabel_value = "Stress Level (%)"
title = "Patient Stress Levels over Time during Therapy Program"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 6))

# Plotting the stress level trend
plt.plot(months, stress_levels, label=label_1, color="#2ca02c", linewidth=2)

# Adding error ellipses to demonstrate variability
for i, (px, py, size) in enumerate(zip(points_months, points_stress_levels, error_size)):
    ellipse = plt.matplotlib.patches.Ellipse(
        (px, py), width=size, height=size/2, color="lightblue", alpha=0.5, edgecolor="black"
    )
    plt.gca().add_patch(ellipse)

# Scatter plot for mean stress level points
plt.scatter(points_months, points_stress_levels, color="blue", marker="o", label=label_3, zorder=5)

# Adding axes labels and title
plt.xlabel(xlabel_value)
plt.ylabel(ylabel_value)
plt.title(title)
plt.ylim(ylim_values)

# Enhancing grid and background
plt.grid(True, which="major", linestyle='--', linewidth='0.5', color='grey')
plt.gca().set_facecolor("white")
plt.gca().set_axisbelow(True)

# Customizing legend
plt.legend(facecolor="white", framealpha=1, edgecolor="black")

# Hiding chart spines and tick marks
for spine in plt.gca().spines.values():
    spine.set_visible(False)
plt.gca().tick_params(axis="both", length=0)  # Hide tick marks

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_36.pdf", bbox_inches="tight")
