

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generating synthetic data for Stress Levels across different age groups
age_groups = np.arange(20, 70, 10)
stress_actual = np.array([4.5, 5.2, 6.0, 6.5, 7.0])
stress_predicted = stress_actual + np.random.normal(0, 0.5, len(stress_actual))

# Error ellipses data (showing variance in prediction)
ellipse_sizes = np.linspace(0.5, 1.5, len(age_groups))

# Labels and Plot Types
label_actual = "Actual Stress Levels"
label_predicted = "Predicted Stress Levels ± 1-σ"
label_points = "Observed Data"
ylim_values = [0, 10]

# Axes Limits and Labels
xlabel_value = "Age Group"
ylabel_value = "Stress Level (0-10)"
title = "Stress Levels Across Different Age Groups"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
plt.figure(figsize=(10, 6))
plt.plot(age_groups, stress_actual, label=label_actual, color="#388e3c", linestyle='-', linewidth=2)
plt.plot(age_groups, stress_predicted, label=label_predicted, color="#fbc02d", linestyle='--', linewidth=2)

# Plot error ellipses
for i, (age, stress, size) in enumerate(zip(age_groups, stress_predicted, ellipse_sizes)):
    ellipse = plt.matplotlib.patches.Ellipse(
        (age, stress), width=2, height=size, edgecolor="#ff7043", facecolor="none"
    )
    plt.gca().add_patch(ellipse)

# Scatter plot of actual data points
plt.scatter(age_groups, stress_actual, color="#303f9f", label=label_points, zorder=5, marker='o')

plt.xlabel(xlabel_value)
plt.ylabel(ylabel_value)
plt.title(title)
plt.ylim(ylim_values)
plt.grid(True, color="white")
plt.gca().set_facecolor("#f5f5f5")
plt.gca().set_axisbelow(True)

# Adjusting the legend
plt.legend(facecolor="#f5f5f5")
for spine in plt.gca().spines.values():
    spine.set_visible(False)
plt.gca().tick_params(axis="both", length=0)  # Hide tick marks

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_37.pdf", bbox_inches="tight")

