
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data
categories = ["Economy Policy", "Healthcare Policy", "Education Policy", "Foreign Policy"]
models = ["Party A", "Party B", "Party C", "Party D", "Party E"]
values = np.random.rand(4, 5) * 50 + 50  # Random values for political domain
referlines = [50, 60, 70, 80, 90]
ylabel = "Success Rate (%)"
xlabel = 'Policy Types'
title = "Comparative Success Rates of Political Parties"
supertitle = "Political Parties Performance Analysis"
ylim = [45, 105]
arrowstart = (0.05, 0.02)
arrowend = (0.48, 0.02)
arrowstart2 = (0.55, 0.02)
arrowend2 = (0.9, 0.02)
xlim = [-0.5, 3.5]
textposition = [[0.5, 50], [2.5, 50]]
textlabel = "Analysis Period: 2021-2023"
spanposition = [[-0.5, 1.5], [1.5, 3.5]]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
colors = ["#2E86C1", "#5DADE2", "#1ABC9C", "#F39C12", "#CB4335"]

# Plotting
fig, ax = plt.subplots(figsize=(12, 5))
width = 0.1
x = np.arange(len(categories))
ax.axvspan(
    spanposition[0][0],
    spanposition[0][1],
    color="#EAF2F8",
)
ax.axvspan(
    spanposition[1][0],
    spanposition[1][1],
    color="#FDEDEC",
)

for i, subcategory in enumerate(categories):
    for j, (model, color, referline) in enumerate(zip(models, colors, referlines)):
        ax.bar(
            i + (j - 2) * width,
            values[i, j] - referline,
            width,
            bottom=referline,
            label=model if i == 0 else "",
            color=color,
        )

# Annotations
for k, model in enumerate(models):
    for i, category in enumerate(categories):
        ax.text(
            i + (k - 2) * width, values[i, k] + 0.5, f"{values[i, k]:.1f}", ha="center"
        )

for line, color in zip(referlines, colors):
    ax.axhline(line, color=color, linestyle="--", linewidth=1.2)
    ax.text(3.4, line + 0.5, f"{line:.1f}", ha="center", color=color)

# Customizations
ax.set_ylabel(ylabel)
ax.set_xlabel(xlabel)
ax.set_xticks(x)
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
plt.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    xycoords="figure fraction",
    arrowprops=dict(arrowstyle="<->", color="green", lw=1),
)
plt.annotate(
    "",
    xy=arrowstart2,
    xytext=arrowend2,
    xycoords="figure fraction",
    arrowprops=dict(arrowstyle="<->", color="purple", lw=1),
)
current_ticks = ax.get_xticks()
new_ticks = current_ticks + 0.5
ax.set_xlim(xlim)
# Set the new ticks without labels
ax.set_xticks(new_ticks, minor=True)  # Add as minor ticks
ax.xaxis.set_minor_formatter(plt.NullFormatter())  # Hide labels for minor ticks

# Enable grid for minor ticks, adjust grid appearance as needed
ax.grid(which="minor", color="gray", linestyle="--", linewidth=0.5, alpha=0.7)
ax.text(
    textposition[0][0],
    textposition[0][1],
    textlabel,
    ha="center",
    va="top",
    fontsize=12,
    color="green",
)
ax.text(
    textposition[1][0],
    textposition[1][1],
    textlabel,
    ha="center",
    va="top",
    fontsize=12,
    color="purple",
)
ax.legend(ncol=5, loc="upper center", bbox_to_anchor=(0.5, 1.2))

# Add main and subtitle
plt.suptitle(supertitle, fontsize=16)
plt.title(title, fontsize=14)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout(rect=[0, 0, 1, 1])
plt.savefig("HR_83.pdf", bbox_inches="tight")
