
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)
# Data
categories = ["Chemotherapy", "Radiation Therapy", "Surgery", "Immunotherapy"]
models = [
    "Effectiveness",
    "Side Effects",
    "Patient Satisfaction",
    "Cost Efficiency",
    "Survival Rate",
]
values = np.random.rand(4, 5) * 40 + 50  # Random values for demonstration, scaled appropriately
referlines = [60, 70, 80, 90, 100]

ylabel = "Health Metric Score"
xlabel = "Type of Treatment"
title = "Comparison of Various Health Treatments"
ylim = [50, 120]
arrowstart = (0.05, 0.02)
arrowend = (0.48, 0.02)
arrowstart2 = (0.55, 0.02)
arrowend2 = (0.9, 0.02)
xlim = [-0.5, 3.5]
textposition = [[0.5, 54], [2.5, 54]]
textlabel = "Health Metrics Analysis"
spanposition = [[-0.5, 1.5], [1.5, 3.5]]
legendlabels = models

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
colors = ["#a8dadc", "#457b9d", "#1d3557", "#06d6a0", "#118ab2"]  # New color scheme
# Plotting
fig, ax = plt.subplots(figsize=(10, 5))
width = 0.1
x = np.arange(len(categories))
ax.axvspan(
    spanposition[0][0],
    spanposition[0][1],
    color="#f0f8ff",
)
ax.axvspan(
    spanposition[1][0],
    spanposition[1][1],
    color="#e0f7fa",
)

for i, subcategory in enumerate(categories):
    for j, (model, color, referline) in enumerate(zip(models, colors, referlines)):
        ax.bar(
            i + (j - 2) * width,
            values[i, j] - referline,
            width,
            bottom=referline,
            label=model if i == 0 else "",
            color=color,
        )

# Annotations
for k, model in enumerate(models):
    for i, category in enumerate(categories):
        ax.text(
            i + (k - 2) * width, values[i, k] + 1, f"{values[i, k]:.1f}", ha="center"
        )

for line, color in zip(referlines, colors):
    ax.axhline(line, color=color, linestyle="--")
    ax.text(3.4, line + 1, f"{line:.1f}", ha="center", color=color)

# Customizations
ax.set_ylabel(ylabel)
ax.set_xlabel(xlabel)
ax.set_title(title)
ax.set_xticks(x)
ax.set_xticklabels(categories)
ax.set_ylim(ylim)
ax.spines["top"].set_visible(False)
ax.spines["right"].set_visible(False)
plt.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    xycoords="figure fraction",
    arrowprops=dict(arrowstyle="<->", color="green", lw=1),
)
plt.annotate(
    "",
    xy=arrowstart2,
    xytext=arrowend2,
    xycoords="figure fraction",
    arrowprops=dict(arrowstyle="<->", color="purple", lw=1),
)
current_ticks = ax.get_xticks()
new_ticks = current_ticks + 0.5
ax.set_xlim(xlim)
# Set the new ticks without labels
ax.set_xticks(new_ticks, minor=True)  # Add as minor ticks
ax.xaxis.set_minor_formatter(plt.NullFormatter())  # Hide labels for minor ticks

# Enable grid for minor ticks, adjust grid appearance as needed
ax.grid(which="minor", color="gray", linestyle="--", alpha=0.5)
ax.text(
    textposition[0][0],
    textposition[0][1]-10,
    textlabel,
    ha="center",
    va="top",
    fontsize=12,
    color="green",
)
ax.text(
    textposition[1][0],
    textposition[1][1]-10,
    textlabel,
    ha="center",
    va="top",
    fontsize=12,
    color="purple",
)
ax.legend(ncol=5, loc="upper center", bbox_to_anchor=(0.5, 1.2), labels=legendlabels)

# ===================
# Part 4: Saving Output
# ===================
plt.tight_layout()
plt.savefig("HR_84.pdf", bbox_inches="tight")
