
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt
import matplotlib.tri as tri

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Define grid parameters
n_angles = 36  # Number of angles
n_radii = 10   # Number of radii (concentric circles)
min_radius = 0.2
radii = np.linspace(min_radius, 1.0, n_radii)

angles = np.linspace(0, 2 * np.pi, n_angles, endpoint=False)
angles = np.repeat(angles[..., np.newaxis], n_radii, axis=1)

x = (radii * np.cos(angles)).flatten()
y = (radii * np.sin(angles)).flatten()

# Generate a Gaussian distribution for BMI values
mean_bmi = 25
std_bmi = 5
bmi_values = np.random.normal(mean_bmi, std_bmi, n_radii * n_angles)
bmi_values = np.clip(bmi_values, 15, 40)  # Clip values to realistic BMI range

# Chart text labels
title = "BMI Distribution Across Regions"
xlabel = "Region X-coordinate"
ylabel = "Region Y-coordinate"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the Triangulation
triang = tri.Triangulation(x, y)

# Mask off unwanted triangles
triang.set_mask(np.hypot(x[triang.triangles].mean(axis=1), 
                         y[triang.triangles].mean(axis=1)) < min_radius)

fig, ax = plt.subplots(figsize=(10, 8))
ax.set_aspect('equal')

# Create tripcolor plot with customized colormap
tpc = ax.tripcolor(triang, bmi_values, shading='flat', cmap='viridis')
fig.colorbar(tpc)

# Additional plot settings
ax.set_title(title, fontsize=14, fontweight='bold')
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.grid(True, linestyle='--', alpha=0.5)

# ===================
# Part 4: Saving Output
# ===================
# Displaying the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("HR_96.pdf", bbox_inches="tight")
