

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt


# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Generating new data for politics domain (e.g., Economic Freedom Score vs. Corruption Perception Index across different regions)
regional_groups = {
    "North America": np.random.normal(loc=(70, 40), scale=10, size=(50, 2)),
    "Europe": np.random.normal(loc=(75, 35), scale=10, size=(50, 2)),
    "Asia": np.random.normal(loc=(60, 50), scale=10, size=(50, 2)),
    "Africa": np.random.normal(loc=(50, 60), scale=10, size=(50, 2)),
    "South America": np.random.normal(loc=(55, 55), scale=10, size=(50, 2)),
    "Oceania": np.random.normal(loc=(80, 30), scale=10, size=(50, 2)),
}

# Text labels and other parameters
insetaxes = [0.15, 0.15, 0.3, 0.3]
insetxlim = [60, 80]
insetylim = [40, 60]
insetxticks = [60,70, 80]
insetyticks = [40,50, 60]
arrowstart = (65, 55)
arrowend = (0.4, 0.4)
annotaterecx = [60, 80]
annotaterecy = [40, 60]
xlabel = 'Economic Freedom Score'
ylabel = 'Corruption Perception Index'
title = 'Political Data: Economic Freedom vs Corruption by Region'
legends = list(regional_groups.keys())

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the scatter plot
# Colors for each regional group (consistent with original color structure)
colors = {
    "North America": "#1f77b4",  # muted blue
    "Europe": "#ff7f0e",  # muted orange
    "Asia": "#2ca02c",  # muted green
    "Africa": "#d62728",  # muted red
    "South America": "#9467bd",  # muted purple
    "Oceania": "#8c564b",  # muted brown
}

fig, ax = plt.subplots(figsize=(8, 8))
for region, data in regional_groups.items():
    ax.scatter(data[:, 0], data[:, 1], label=region, c=colors[region], alpha=0.6, edgecolors='w', s=100)

ax.set_xlabel(xlabel)
ax.set_ylabel(ylabel)
ax.set_title(title)
ax.legend(title='Regional Groups', loc='upper left')

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(insetaxes)  # Adjust the position to align with the right side of the main plot
for region, data in regional_groups.items():
    ax_inset.scatter(data[:, 0], data[:, 1], c=colors[region], alpha=0.6, edgecolors='w', s=100)
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Drawing rectangle annotation
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[1], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[0]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[0], annotaterecx[0]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)
ax.plot(
    [annotaterecx[1], annotaterecx[1]],
    [annotaterecy[0], annotaterecy[1]],
    color="black",
    lw=1,
)

# Annotation with arrow
ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.5),
)

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_46.pdf", bbox_inches="tight")
