

# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)

# Updated time steps representing days
time_steps = np.linspace(0, 200, 50)

# Generate different trends for each line
trend1_values = np.cumsum(np.random.normal(200, 50, size=len(time_steps)))  # Linear growth with noise (daily movie viewers)
trend2_values = 1000 + 300 * np.sin(time_steps * 0.2) + np.random.normal(0, 50, len(time_steps))  # Periodic spikes in ticket sales
trend3_values = np.random.normal(500, 200, size=len(time_steps))  # Random fluctuations in social media mentions
trend4_values = np.log(time_steps + 1) * 1000  # Logarithmic growth in box office revenue

# Simulate standard deviations for error
std_dev = 100.0
trend1_std = np.full_like(trend1_values, std_dev)
trend2_std = np.full_like(trend2_values, std_dev)
trend3_std = np.full_like(trend3_values, std_dev)
trend4_std = np.full_like(trend4_values, std_dev)

# Text labels and axes settings
xlabel = "Time (Days)"
ylabel = "Metrics"
labels = ["Trend 3 (Social Media Mentions)", "Trend 4 (Box Office Revenue)"]
title = "Movie Industry Trends Over Time"
supertitle = "Movie Metrics Analysis"
legend_labels = ["Social Media Mentions", "Box Office Revenue"]
xticks = np.linspace(0, 200, 9)
yticks = np.arange(0, 10000, 1000)
axesinset = [0.7, 0.35, 0.23, 0.2]
insetxlim = [25, 75]
insetylim = [0, 8000]
insetxticks = [25, 50, 75]
insetyticks = [0, 4000, 8000]
arrowstart = (75, 2000)
arrowend = (0.7, 0.35)
annotaterecx = [25, 75]
annotaterecy = [0, 4000]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a single plot
fig, ax = plt.subplots(figsize=(8, 6))

# Plot the third trend on the main plot
ax.plot(time_steps, trend3_values, "o--", color="blue", label=legend_labels[0])
ax.fill_between(
    time_steps, trend3_values - trend3_std, trend3_values + trend3_std, color="blue", alpha=0.2
)

# Plot the fourth trend on the main plot
ax.plot(time_steps, trend4_values, "s-", color="purple", label=legend_labels[1])
ax.fill_between(
    time_steps, trend4_values - trend4_std, trend4_values + trend4_std, color="purple", alpha=0.2
)

# Set labels, ticks, legend, and grid for the main plot
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_xticks(xticks)
ax.set_yticks(yticks)
ax.legend(loc="upper left", shadow=True, frameon=True, framealpha=0.9)
ax.grid(True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5)
ax.set_facecolor("#f5f5f5")
ax.set_title(title, fontsize=14)
fig.suptitle(supertitle, fontsize=16, fontweight='bold')

# Draw a rectangle on the main plot to indicate the area of zoom-in
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[1], annotaterecy[1]], color="black", lw=1)
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[0], annotaterecy[0]], color="black", lw=1)
ax.plot([annotaterecx[0], annotaterecx[0]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1)
ax.plot([annotaterecx[1], annotaterecx[1]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(axesinset)

# Plot the third trend on the inset
ax_inset.plot(time_steps, trend3_values, "o--", color="blue", label=legend_labels[0])
ax_inset.fill_between(
    time_steps, trend3_values - trend3_std, trend3_values + trend3_std, color="blue", alpha=0.2
)

# Plot the fourth trend on the inset
ax_inset.plot(time_steps, trend4_values, "s-", color="purple", label=legend_labels[1])
ax_inset.fill_between(
    time_steps, trend4_values - trend4_std, trend4_values + trend4_std, color="purple", alpha=0.2
)

# Set limits, ticks and border color for the inset
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add an arrow from the rectangle on the main plot to the inset
ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=0.1),
)

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("PIP_48.pdf", bbox_inches="tight")

