
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(0)

# Updated gradient steps for simulated time (weeks)
weeks = np.linspace(0, 50, 50)

# Generate different performance metrics for each line
speed = np.linspace(30, 80, len(weeks)) + np.random.normal(0, 5, len(weeks))  # Linear growth with noise
stamina = np.exp(0.05 * weeks)  # Exponential growth
agility = np.log(weeks + 1) * 10  # Logarithmic trend
strength = 2 * weeks  # Linear growth

# Simulate standard deviations for error bars
std_dev = 5
std_speed = np.full_like(speed, std_dev)
std_stamina = np.full_like(stamina, std_dev)
std_agility = np.full_like(agility, std_dev)
std_strength = np.full_like(strength, std_dev)

# Labels and other text parameters
xlabel = "Weeks"
ylabel = "Performance Metric"
title = "Athlete Performance Metrics Over a Season"
legend_labels = ["Speed", "Stamina", "Agility", "Strength"]
xticks = np.linspace(0, 50, 11)
yticks = np.arange(0, 110, 10)
axesinset = [0.16, 0.6, 0.2, 0.2]
insetxlim = [10, 40]
insetylim = [20, 80]
insetxticks = [20, 30, 40]
insetyticks = [40, 60, 80]
arrowstart = (20, 60)
arrowend = (0.2, 0.7)
annotaterecx = [20, 40]
annotaterecy = [30, 80]

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create a figure with a single plot
fig, ax = plt.subplots(figsize=(10, 6))

# Plot the first metric (speed) on the main plot
ax.plot(weeks, speed, 'o--', color='#FF7F0E', label=legend_labels[0])
ax.fill_between(weeks, speed - std_speed, speed + std_speed, color='#FF7F0E', alpha=0.2)

# Plot the second metric (stamina) on the main plot
ax.plot(weeks, stamina, 's-', color='#1F77B4', label=legend_labels[1])
ax.fill_between(weeks, stamina - std_stamina, stamina + std_stamina, color='#1F77B4', alpha=0.2)

# Additional plots - agility and strength
ax.plot(weeks, agility, 'v-.', color='#2CA02C', label="Agility")
ax.fill_between(weeks, agility - std_agility, agility + std_agility, color='#2CA02C', alpha=0.2)

ax.plot(weeks, strength, 'd:', color='#9467BD', label="Strength")
ax.fill_between(weeks, strength - std_strength, strength + std_strength, color='#9467BD', alpha=0.2)

# Set labels, title, ticks, and grid for the main plot
ax.set_xlabel(xlabel, fontsize=12)
ax.set_ylabel(ylabel, fontsize=12)
ax.set_title(title, fontsize=14)
ax.set_xticks(xticks)
ax.set_yticks(yticks)
ax.legend(loc="upper center",ncol=4,frameon=False, framealpha=0.9,bbox_to_anchor=(0.5, 1.2))
ax.grid(True, which="both", axis="both", color="lightgray", linestyle="--", linewidth=0.5)
ax.set_facecolor("#f9f9f9")

# Draw a rectangle on the main plot to indicate the area of zoom-in
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[1], annotaterecy[1]], color="black", lw=1)
ax.plot([annotaterecx[0], annotaterecx[1]], [annotaterecy[0], annotaterecy[0]], color="black", lw=1)
ax.plot([annotaterecx[0], annotaterecx[0]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1)
ax.plot([annotaterecx[1], annotaterecx[1]], [annotaterecy[0], annotaterecy[1]], color="black", lw=1)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(axesinset)  # Adjust the position to align with the right side of the main plot

# Plot the speed metric on the inset
ax_inset.plot(weeks, speed, 'o--', color='#FF7F0E', label=legend_labels[0])
ax_inset.fill_between(weeks, speed - std_speed, speed + std_speed, color='#FF7F0E', alpha=0.2)

# Plot the stamina metric on the inset
ax_inset.plot(weeks, stamina, 's-', color='#1F77B4', label=legend_labels[1])
ax_inset.fill_between(weeks, stamina - std_stamina, stamina + std_stamina, color='#1F77B4', alpha=0.2)

# Set limits, ticks and border color for the inset
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add an arrow from the rectangle on the main plot to the inset
ax.annotate("", xy=arrowstart, xytext=arrowend, textcoords="axes fraction", arrowprops=dict(facecolor="black", lw=0.1))

# ===================
# Part 4: Saving Output
# ===================
# Adjust layout and display the plot
plt.tight_layout()
plt.savefig("PIP_50.pdf", bbox_inches="tight")
