
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

# Simulated data for weekly visitors (in thousands) to four different art exhibitions over a 52-week period
np.random.seed(0)
weeks = np.linspace(1, 52, 52)
visitors_exhibition1 = 50 + 10 * np.sin(2 * np.pi * weeks / 52) + np.cumsum(np.random.normal(0, 1, 52))
visitors_exhibition2 = 55 + 12 * np.sin(2 * np.pi * weeks / 52) + np.cumsum(np.random.normal(0, 1.2, 52))
visitors_exhibition3 = 45 + 8 * np.sin(2 * np.pi * weeks / 52) + np.cumsum(np.random.normal(0, 0.9, 52))
visitors_exhibition4 = 48 + 11 * np.sin(2 * np.pi * weeks / 52) + np.cumsum(np.random.normal(0, 1.1, 52))

# Labels and Plot Types
label_exhibition1 = "Exhibition A"
label_exhibition2 = "Exhibition B"
label_exhibition3 = "Exhibition C"
label_exhibition4 = "Exhibition D"

# Axes Limits and Labels
xlabel_value = "Weeks"
ylabel_value = "Number of Visitors (Thousands)"
zoomed_in_axes = [0.15, 0.15, 0.25, 0.25]
xlim_values = [47, 52]
ylim_values = [
    min(
        visitors_exhibition1[46],
        visitors_exhibition2[46],
        visitors_exhibition3[46],
        visitors_exhibition4[46],
    ),
    max(
        visitors_exhibition1[51],
        visitors_exhibition2[51],
        visitors_exhibition3[51],
        visitors_exhibition4[51],
    )
    + 5,
]
xticks_values = list(range(47, 53))
yticks_values = list(range(int(ylim_values[0]), int(ylim_values[1]) + 5, 5))
title = "Weekly Visitors to Art Exhibitions Over 52 Weeks"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Create the main figure and axis
fig, ax = plt.subplots(figsize=(12, 7))

# Plot the curves with distinct styles
ax.plot(
    weeks,
    visitors_exhibition1,
    color="#1f77b4",
    linestyle="-",
    marker="o",
    label=label_exhibition1,
)
ax.plot(
    weeks,
    visitors_exhibition2,
    color="#ff7f0e",
    linestyle="--",
    marker="s",
    label=label_exhibition2,
)
ax.plot(
    weeks,
    visitors_exhibition3,
    color="#2ca02c",
    linestyle="-.",
    marker="^",
    label=label_exhibition3,
)
ax.plot(
    weeks,
    visitors_exhibition4,
    color="#d62728",
    linestyle=":",
    marker="d",
    label=label_exhibition4,
)

# Set labels and title
ax.set_xlabel(xlabel_value)
ax.set_ylabel(ylabel_value)
ax.set_title(title)

# Create the inset with the zoomed-in view
ax_inset = fig.add_axes(zoomed_in_axes)
ax_inset.plot(weeks, visitors_exhibition1, color="#1f77b4", linestyle="-", marker="o")
ax_inset.plot(weeks, visitors_exhibition2, color="#ff7f0e", linestyle="--", marker="s")
ax_inset.plot(weeks, visitors_exhibition3, color="#2ca02c", linestyle="-.", marker="^")
ax_inset.plot(weeks, visitors_exhibition4, color="#d62728", linestyle=":", marker="d")
ax_inset.set_xlim(xlim_values)
ax_inset.set_xticks(xticks_values)
ax_inset.spines["bottom"].set_color("black")
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

# Add the legend to the main axis, outside the plot area
ax.legend(loc="upper left")

# ===================
# Part 4: Saving Output
# ===================
# Show the plot with tight layout to minimize white space
plt.tight_layout()
plt.savefig("PIP_73.pdf", bbox_inches="tight")
