
# ===================
# Part 1: Importing Libraries
# ===================
import matplotlib.pyplot as plt

# ===================
# Part 2: Data Preparation
# ===================
import numpy as np

np.random.seed(42)
# Data for energy sources
energy_sources = {
    "Solar": np.random.normal(loc=(150, 90), scale=20, size=(50, 2)),
    "Wind": np.random.normal(loc=(200, 85), scale=25, size=(45, 2)),
    "Hydro": np.random.normal(loc=(180, 92), scale=15, size=(60, 2)),
    "Nuclear": np.random.normal(loc=(250, 98), scale=30, size=(55, 2)),
    "Coal": np.random.normal(loc=(230, 70), scale=20, size=(35, 2)),
    "NaturalGas": np.random.normal(loc=(210, 80), scale=25, size=(50, 2)),
}

title = "Energy Source Performance Across Daily Consumption and Efficiency"
xlabel = "Average Daily Consumption (MWh)"
ylabel = "Efficiency Score"
# Inset plot configuration
insetaxes = [0.15, 0.15, 0.2, 0.2]
insetxlim = [140, 200]  # Focus on Solar and Hydro
insetylim = [80, 100]
insetxticks = [140, 160, 180, 200]
insetyticks = [80, 85, 90, 95, 100]

# Arrow and annotation configuration
arrowstart = (150, 90)  # Arrow start for Solar
arrowend = (0.24, 0.32)  # Relative inset arrow end
annotaterecx = [140, 200]  # Annotation range for X
annotaterecy = [80, 100]  # Annotation range for Y
legendtitle="Energy Sources"

# ===================
# Part 3: Plot Configuration and Rendering
# ===================
# Colors for energy sources
colors = {
    "Solar": "#FF5733",
    "Wind": "#33FF57",
    "Hydro": "#3357FF",
    "Nuclear": "#FF33A1",
    "Coal": "#9333FF",
    "NaturalGas": "#33FFF6",
}

# Create the scatter plot
fig, ax = plt.subplots(figsize=(10, 8))
for source, data in energy_sources.items():
    ax.scatter(
        data[:, 0],
        data[:, 1],
        c=colors[source],
        label=source,
        alpha=0.6,
        edgecolors="w",
        s=100,
        marker="o",  # Circle marker for better visibility
    )

# Enclosing rectangle for annotation
rect = plt.Rectangle(
    (annotaterecx[0], annotaterecy[0]),
    annotaterecx[1] - annotaterecx[0],
    annotaterecy[1] - annotaterecy[0],
    linewidth=1,
    edgecolor="black",
    facecolor="none",
    linestyle="--",  # Dashed line for rectangle
)
ax.add_patch(rect)

# Create the inset with zoomed-in view
ax_inset = fig.add_axes(insetaxes)
for source, data in energy_sources.items():
    ax_inset.scatter(
        data[:, 0], data[:, 1], c=colors[source], alpha=0.6, edgecolors="w", s=100
    )
ax_inset.set_xlim(insetxlim)
ax_inset.set_ylim(insetylim)
ax_inset.set_xticks(insetxticks)
ax_inset.set_yticks(insetyticks)
ax_inset.spines["bottom"].set_color("black")  # Add black border to the inset
ax_inset.spines["left"].set_color("black")
ax_inset.spines["top"].set_color("black")
ax_inset.spines["right"].set_color("black")

ax.annotate(
    "",
    xy=arrowstart,
    xytext=arrowend,
    textcoords="axes fraction",
    arrowprops=dict(facecolor="black", lw=1),
)

# Adding titles and labels
ax.set_title(title, fontsize=16)
ax.set_xlabel(xlabel, fontsize=14)
ax.set_ylabel(ylabel, fontsize=14)
ax.legend(title=legendtitle, title_fontsize="13")

# ===================
# Part 4: Saving Output
# ===================
# Show the plot
plt.tight_layout()
plt.savefig("PIP_77.pdf", bbox_inches="tight")
